<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\SellTransaction;
use Carbon\Carbon;
use Illuminate\Http\Request;

class SellTransactionController extends Controller
{

    public function index(Request $request)
    {
        $query = SellTransaction::with('product');

        // Apply Date Filter
        if ($request->has('date_range') && !empty($request->date_range)) {
            $dates = explode(' - ', $request->date_range);
            $startDate = date('Y-m-d', strtotime($dates[0]));
            $endDate = date('Y-m-d', strtotime($dates[1]));

            $query->whereBetween('created_at', [$startDate, $endDate]);
        }

        $sales = $query->orderBy('created_at', 'desc')->get();

        $totals = [
            'total_profit' => $sales->sum('profit'),
            'total_sale' => $sales->sum('sold_quantity'),
            'total_amount' => $sales->sum(fn($sale) => $sale->sell_price * $sale->sold_quantity)
        ];

        return view('sell.index', compact('sales', 'totals'));
    }

    public function showSaleForm($id)
    {
        $product = Product::with(['ProductBrand', 'ProductModel', 'Customer'])->findOrFail($id);
        return view('sell.form', compact('product'));
    }

    public function profitReport($id)
    {
        $product = Product::with(['ProductBrand', 'ProductModel', 'Customer'])->findOrFail($id);
        return view('sell.form', compact('product'));
    }

    public function processSale(Request $request, $id)
    {
        $request->validate([
            'customer_name' => 'required|string',
            // 'customer_mobile' => 'nullable|string',
            'sell_price' => 'required|numeric|min:0',
            'sold_quantity' => 'required|integer|min:1',
            'warranty' => 'required|string',
        ]);
        // return $request;

        $product = Product::with(['ProductBrand', 'ProductModel', 'Customer'])->findOrFail($id);

        if ($request->sold_quantity > $product->stock) {
            return back()->with('error', 'Not enough stock available. Available stock: ' . $product->stock);
        }

        $profit = ($request->sell_price - $product->buy_price) * $request->sold_quantity;

        $warranty_duration = $request->warranty;
        if ($request->warranty == 'Custom') {
            $warranty_duration = $request->custom_warranty_value . ' ' . $request->custom_warranty_unit;
        }

        $sale = SellTransaction::create([
            'product_id' => $product->id,
            'customer_name' => $request->customer_name,
            'customer_mobile' => $request->customer_mobile,
            'sell_price' => $request->sell_price,
            'sold_quantity' => $request->sold_quantity,
            'profit' => $profit,
            'warranty' => $warranty_duration,
            'date_time' => now(),
            'brand' => $product->brand,
            'model' => $product->model,
        ]);

        $product->decrement('stock', $request->sold_quantity);

        return view('sell.invoice', compact(['sale', 'product']))->with('success', 'Sale recorded successfully!');
    }

    public function saleInvoice($id)
    {
        $sale = SellTransaction::with('product')->findOrFail($id);
        return view('sell.sale_invoice', compact('sale'));
    }

    public function salesProfitReport(Request $request)
    {
        $transactions = SellTransaction::with([
            'product' => function ($query) {
                $query->with(['ProductBrand', 'ProductModel']);
            }
        ]);

        // If it's an AJAX request with a date range
        if ($request->ajax() && $request->has('date_range')) {
            $dates = explode(' - ', $request->date_range);
            $startDate = Carbon::parse($dates[0])->startOfDay();
            $endDate = Carbon::parse($dates[1])->endOfDay();

            // Filter transactions within the selected date range
            $transactions->whereBetween('created_at', [$startDate, $endDate]);
        }

        $transactions = $transactions->get(); // Execute query

        if ($request->ajax()) {
            return response()->json(['transactions' => $transactions]);
        }

        return view('sell.profit_report', compact('transactions'));
    }



    public function destroy($sellTransaction)
    {
        // Find the transaction
        $sellTransaction = SellTransaction::findOrFail($sellTransaction);

        // Find the product related to the transaction
        $soldProduct = Product::findOrFail($sellTransaction->product_id);

        // Calculate the total quantity to restore in the stock
        $total_qty = $soldProduct->stock + $sellTransaction->sold_quantity;

        // Try to delete the transaction
        if ($sellTransaction->delete()) {
            // Only update the product stock if the deletion was successful
            $soldProduct->stock = $total_qty;
            $soldProduct->save();

            // Redirect with success message
            return redirect()->route('sale.index')->with('success', 'Successfully deleted!');
        } else {
            // In case the deletion fails
            return redirect()->route('sale.index')->with('error', 'Failed to delete transaction!');
        }
    }
}
