<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use Illuminate\Http\Request;

class BrandController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $tenant = app('tenant');
        $brands = Brand::where('tenant_id', $tenant->id)->get();
        return view('admin.brands.index', compact('brands'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.brands.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Validate request
            $validatedData = $request->validate([
                'brand_name' => 'required|string|max:20'
            ]);

            $tenant = app('tenant');

            // Insert customer into the database
            $brand = Brand::create([
                'name'    => $validatedData['brand_name'],
                'tenant_id' => $tenant->id,
            ]);

            // Prepare response data
            $brandInfo = [
                'name'   => $brand->name,
            ];

            // Return success message
            return redirect()->route('brands.index')->with('success', 'Brand added successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Handle validation errors
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            // Handle all other exceptions
            return redirect()->back()->with('error', 'An error occurred while adding the brand');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Brand $brand)
    {
        $tenant = app('tenant');
        $brand = Brand::where('tenant_id', $tenant->id)->findOrFail($brand->id);
        return view('admin.brands.show', compact('brand'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Brand $brand)
    {
        $tenant = app('tenant');
        $brand = Brand::where('tenant_id', $tenant->id)->findOrFail($brand->id);
        return view('admin.brands.edit', compact('brand'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Brand $brand)
    {
        try {
            $tenant = app('tenant');
            $brand = Brand::where('tenant_id', $tenant->id)->findOrFail($brand->id);

            // Validate request
            $validatedData = $request->validate([
                'brand_name' => 'required|string|max:20'
            ]);

            // Update brand
            $brand->update([
                'name' => $validatedData['brand_name'],
            ]);

            return redirect()->route('brands.index')->with('success', 'Brand updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while updating the brand');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Brand $brand)
    {
        try {
            $tenant = app('tenant');
            $brand = Brand::where('tenant_id', $tenant->id)->findOrFail($brand->id);
            $brand->delete();
            return redirect()->route('brands.index')->with('success', 'Brand deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while deleting the brand');
        }
    }
}
