<?php

// app/Http/Controllers/ExpenseController.php

namespace App\Http\Controllers;

use App\Models\Expense;
use Illuminate\Http\Request;

class ExpenseController extends Controller
{
    public function create()
    {
        return view('expenses.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'expense_date' => 'required|date',
            'category_name' => 'required|string|max:255',
            'expense_for' => 'required|string|max:255',
            'amount' => 'required|numeric',
        ]);

        Expense::create([
            'expense_date' => $validated['expense_date'],
            'category_name' => $validated['category_name'],
            'expense_for' => $validated['expense_for'],
            'amount' => $validated['amount'],
            'note' => $request->note,
            'tenant_id' => auth()->user()->tenant_id
        ]);

        return redirect()->route('expenses.create')->with('success', 'Expense added successfully!');
    }

    public function report(Request $request)
    {
        $query = Expense::where('tenant_id', auth()->user()->tenant_id);

        if ($request->filled('from_date')) {
            $query->where('expense_date', '>=', $request->from_date);
        }

        if ($request->filled('to_date')) {
            $query->where('expense_date', '<=', $request->to_date);
        }

        if ($request->filled('category_name')) {
            $query->where('category_name', 'like', '%' . $request->category_name . '%');
        }

        $expenses = $query->get();

        $totalExpenses = $expenses->sum('amount');
        $totalCategories = $expenses->groupBy('category_name')->count();
        $chartLabels = $expenses->pluck('category_name')->unique();
        $chartData = $expenses->groupBy('category_name')->map->sum('amount');

        return view('expenses.report', compact('expenses', 'totalExpenses', 'totalCategories', 'chartLabels', 'chartData'));
    }

    public function destroy($id)
    {
        $expense = Expense::where('tenant_id', auth()->user()->tenant_id)->findOrFail($id);
        $expense->delete();

        return redirect()->route('expenses.report')->with('success', 'Expense deleted successfully!');
    }
}
