<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Auth\Events\Registered;

class RegisterController extends Controller
{
    public function showRegistrationForm()
    {
        return view('auth.register');
    }

    public function register(Request $request)
    {
        $this->validator($request->all())->validate();

        // Create tenant for the user (or assign existing tenant logic)
        $tenant = Tenant::create([
            'name' => $request->input('tenant_name'),
            'domain' => $request->input('tenant_domain'),
            'subdomain' => $request->input('tenant_subdomain'),
        ]);

        $user = $this->create($request->all(), $tenant->id);

        event(new Registered($user));

        auth()->login($user);

        return redirect()->route('subscriptions.create');
    }

    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'tenant_name' => ['required', 'string', 'max:255'],
            'tenant_domain' => ['required', 'string', 'max:255', 'unique:tenants,domain'],
            'tenant_subdomain' => ['nullable', 'string', 'max:255', 'unique:tenants,subdomain'],
        ]);
    }

    protected function create(array $data, $tenantId)
    {
        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
            'tenant_id' => $tenantId,
        ]);

        // Assign default role to user
        // Assign default role 'admin' for the first user or 'user' for others
        if (\Spatie\Permission\Models\Role::where('name', 'admin')->exists()) {
            $user->assignRole('admin');
        } elseif (\Spatie\Permission\Models\Role::where('name', 'user')->exists()) {
            $user->assignRole('user');
        }

        return $user;
    }
}
