<?php

namespace App\Http\Controllers;

use App\Models\MobileModel;
use Illuminate\Http\Request;

class MobileModelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $tenant = app('tenant');
        $models = MobileModel::with('brand')->where('tenant_id', $tenant->id)->get();
        return view('admin.mobile-models.index', compact('models'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $tenant = app('tenant');
        $brands = \App\Models\Brand::where('tenant_id', $tenant->id)->get();
        return view('admin.mobile-models.create', compact('brands'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Validate request
            $validatedData = $request->validate([
                'brand_id'   => 'required|integer|exists:brands,id',
                'model_name' => 'required|string|max:20'
            ]);

            // return $validatedData;
            $tenant = app('tenant');

            // Insert model into the database
            $model = MobileModel::create([
                'tenant_id' => $tenant->id,
                'brand_id' => $validatedData['brand_id'], // Ensure brand_id is stored
                'name'     => $validatedData['model_name'],
            ]);

            // Prepare response data
            return redirect()->route('mobile-models.index')->with('success', 'Model added successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while adding the model');
        }
    }


    /**
     * Display the specified resource.
     */
    public function show(MobileModel $mobileModel)
    {
        $mobileModel->load('brand');
        return view('admin.mobile-models.show', compact('mobileModel'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MobileModel $mobileModel)
    {
        $tenant = app('tenant');
        $brands = \App\Models\Brand::where('tenant_id', $tenant->id)->get();
        return view('admin.mobile-models.edit', compact('mobileModel', 'brands'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MobileModel $mobileModel)
    {
        try {
            $validatedData = $request->validate([
                'brand_id'   => 'required|integer|exists:brands,id',
                'model_name' => 'required|string|max:20'
            ]);

            $mobileModel->update([
                'brand_id' => $validatedData['brand_id'],
                'name'     => $validatedData['model_name'],
            ]);

            return redirect()->route('mobile-models.index')->with('success', 'Model updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while updating the model');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MobileModel $mobileModel)
    {
        try {
            $mobileModel->delete();
            return redirect()->route('mobile-models.index')->with('success', 'Model deleted successfully');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while deleting the model');
        }
    }

    public function getModels($brandId)
    {
        try {
            $tenant = app('tenant');
            $models = MobileModel::where('tenant_id', $tenant->id)->where('brand_id', $brandId)->get();

            return response()->json([
                'success' => true,
                'models' => $models
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching models: ' . $e->getMessage()
            ], 500);
        }
    }
}
