<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use ZipArchive;

class SystemUpdateController extends Controller
{
    // Current system version
    const CURRENT_VERSION = '1.0.0';
    
    // Update server URL (you can change this to your actual update server)
    const UPDATE_SERVER_URL = 'https://yourdomain.com/updates';
    
    /**
     * Display the system update page
     */
    public function index()
    {
        $currentVersion = self::CURRENT_VERSION;
        $latestVersion = $this->checkForUpdates();
        $updateAvailable = version_compare($latestVersion, $currentVersion, '>');
        
        return view('system.update', compact('currentVersion', 'latestVersion', 'updateAvailable'));
    }
    
    /**
     * Check for available updates
     */
    public function checkForUpdates()
    {
        try {
            // In real scenario, this would check your update server
            // For now, return a dummy version
            // You can implement API call to your update server here
            
            // Example: $response = Http::get(self::UPDATE_SERVER_URL . '/latest-version.json');
            // return $response->json()['version'];
            
            return '1.0.1'; // Dummy latest version
        } catch (\Exception $e) {
            Log::error('Update check failed: ' . $e->getMessage());
            return self::CURRENT_VERSION;
        }
    }
    
    /**
     * Perform system update
     */
    public function update(Request $request)
    {
        try {
            $steps = [];
            
            // Step 1: Put application in maintenance mode
            $steps[] = $this->maintenanceMode(true);
            
            // Step 2: Backup current system
            $steps[] = $this->backupSystem();
            
            // Step 3: Download update package
            $updateFile = $this->downloadUpdate();
            $steps[] = ['step' => 'Download Update', 'status' => 'success', 'message' => 'Update package downloaded'];
            
            // Step 4: Extract update files
            $steps[] = $this->extractUpdate($updateFile);
            
            // Step 5: Run database migrations
            $steps[] = $this->runMigrations();
            
            // Step 6: Clear cache
            $steps[] = $this->clearCache();
            
            // Step 7: Turn off maintenance mode
            $steps[] = $this->maintenanceMode(false);
            
            return response()->json([
                'success' => true,
                'message' => 'System updated successfully to version ' . $this->checkForUpdates(),
                'steps' => $steps
            ]);
            
        } catch (\Exception $e) {
            // Restore from backup if update fails
            $this->restoreBackup();
            $this->maintenanceMode(false);
            
            Log::error('System update failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Update failed: ' . $e->getMessage(),
                'steps' => $steps ?? []
            ], 500);
        }
    }
    
    /**
     * Enable/disable maintenance mode
     */
    private function maintenanceMode($enable = true)
    {
        try {
            if ($enable) {
                Artisan::call('down', ['--render' => 'errors::503']);
                return ['step' => 'Maintenance Mode', 'status' => 'success', 'message' => 'System put in maintenance mode'];
            } else {
                Artisan::call('up');
                return ['step' => 'Maintenance Mode Off', 'status' => 'success', 'message' => 'System is back online'];
            }
        } catch (\Exception $e) {
            return ['step' => 'Maintenance Mode', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Backup current system
     */
    private function backupSystem()
    {
        try {
            $backupPath = storage_path('backups');
            if (!File::exists($backupPath)) {
                File::makeDirectory($backupPath, 0755, true);
            }
            
            $backupFile = $backupPath . '/backup_' . date('Y-m-d_H-i-s') . '.zip';
            
            // Create backup zip (simplified version)
            // In production, you'd want to backup database and files
            $zip = new ZipArchive();
            if ($zip->open($backupFile, ZipArchive::CREATE) === true) {
                // Add important files to backup
                $this->addFilesToZip($zip, base_path('app'), 'app');
                $this->addFilesToZip($zip, base_path('config'), 'config');
                $this->addFilesToZip($zip, base_path('database'), 'database');
                $this->addFilesToZip($zip, base_path('resources'), 'resources');
                $this->addFilesToZip($zip, base_path('routes'), 'routes');
                
                $zip->close();
            }
            
            return ['step' => 'Backup', 'status' => 'success', 'message' => 'System backed up successfully'];
        } catch (\Exception $e) {
            return ['step' => 'Backup', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Add files to zip recursively
     */
    private function addFilesToZip($zip, $folder, $zipFolder = '')
    {
        if (!File::exists($folder)) {
            return;
        }
        
        $files = File::allFiles($folder);
        foreach ($files as $file) {
            $relativePath = $zipFolder . '/' . $file->getRelativePathname();
            $zip->addFile($file->getRealPath(), $relativePath);
        }
    }
    
    /**
     * Download update package
     */
    private function downloadUpdate()
    {
        // In real scenario, download from update server
        // For now, return a dummy path
        // Example: 
        // $updateUrl = self::UPDATE_SERVER_URL . '/update-' . $latestVersion . '.zip';
        // $updateFile = storage_path('updates/update.zip');
        // file_put_contents($updateFile, file_get_contents($updateUrl));
        
        return storage_path('updates/update.zip');
    }
    
    /**
     * Extract update files
     */
    private function extractUpdate($updateFile)
    {
        try {
            if (!File::exists($updateFile)) {
                throw new \Exception('Update file not found');
            }
            
            $zip = new ZipArchive();
            if ($zip->open($updateFile) === true) {
                $zip->extractTo(base_path());
                $zip->close();
                
                // Delete update file after extraction
                File::delete($updateFile);
                
                return ['step' => 'Extract Files', 'status' => 'success', 'message' => 'Update files extracted'];
            } else {
                throw new \Exception('Failed to open update package');
            }
        } catch (\Exception $e) {
            return ['step' => 'Extract Files', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Run database migrations
     */
    private function runMigrations()
    {
        try {
            Artisan::call('migrate', ['--force' => true]);
            return ['step' => 'Database Migration', 'status' => 'success', 'message' => 'Database updated successfully'];
        } catch (\Exception $e) {
            return ['step' => 'Database Migration', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Clear system cache
     */
    private function clearCache()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            
            return ['step' => 'Clear Cache', 'status' => 'success', 'message' => 'System cache cleared'];
        } catch (\Exception $e) {
            return ['step' => 'Clear Cache', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Restore from backup
     */
    private function restoreBackup()
    {
        try {
            $backupPath = storage_path('backups');
            $backups = File::files($backupPath);
            
            if (empty($backups)) {
                return false;
            }
            
            // Get the latest backup
            usort($backups, function($a, $b) {
                return $b->getMTime() - $a->getMTime();
            });
            
            $latestBackup = $backups[0];
            
            $zip = new ZipArchive();
            if ($zip->open($latestBackup->getRealPath()) === true) {
                $zip->extractTo(base_path());
                $zip->close();
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            Log::error('Backup restore failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get update history/changelog
     */
    public function changelog()
    {
        $changelog = [
            [
                'version' => '1.0.1',
                'date' => '2025-10-21',
                'changes' => [
                    'Added enhanced notification system with sounds',
                    'Improved sales history delete functionality',
                    'Added auto-update system',
                    'Bug fixes and performance improvements'
                ]
            ],
            [
                'version' => '1.0.0',
                'date' => '2025-10-01',
                'changes' => [
                    'Initial release',
                    'POS system with sales management',
                    'Customer management',
                    'Inventory tracking'
                ]
            ]
        ];
        
        return response()->json($changelog);
    }
}
