<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Customer extends Model
{
    protected $fillable = [
        'name',
        'phone',
        'email',
        'cpr',
        'address',
        'tenant_id',
    ];

    // Relationship with the Sale model (a customer has many sales)
    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    // Calculate total due amount for this customer (after payments)
    public function getTotalDueAmount()
    {
        // Exclude returned sales from due amount calculation
        $totalDueFromSales = $this->sales()
            ->where('return_status', '!=', 'returned')
            ->sum('due_amount');
        $advances = \App\Models\CustomerPayment::where('customer_id', $this->id)
            ->where('type', 'advance')
            ->sum('amount');
        
        // Only subtract advances since due_amount in sales already reflects payments made
        $netDue = $totalDueFromSales - $advances;
        return max(0, $netDue);
    }

    // Get all unpaid sales for this customer (excluding returned)
    public function getUnpaidSales()
    {
        return $this->sales()
            ->where('due_amount', '>', 0)
            ->where('return_status', '!=', 'returned')
            ->get();
    }

    protected static function booted()
    {
        static::addGlobalScope('tenant', function ($builder) {
            if (app()->has('tenant')) {
                $builder->where('tenant_id', app('tenant')->id);
            }
        });
    }
}
