@extends('layouts.admin')

@section('title', 'WhatsApp Web Connection')

@section('content')
<div class="content-wrapper">
    <!-- Content Header -->
    <div class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1 class="m-0">
                        <i class="fab fa-whatsapp mr-2 text-success"></i>WhatsApp Web Connection
                    </h1>
                </div>
                <div class="col-sm-6">
                    <ol class="breadcrumb float-sm-right">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">Home</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('invoice-settings.index') }}">Invoice Settings</a></li>
                        <li class="breadcrumb-item active">WhatsApp Connection</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <!-- Main content -->
    <section class="content">
        <div class="container-fluid">
            
            <!-- Connection Status Card -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card card-outline card-success">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-wifi mr-2"></i>Connection Status
                            </h3>
                            <div class="card-tools">
                                <span id="connectionStatus" class="badge badge-secondary">
                                    <i class="fas fa-circle mr-1"></i>Checking...
                                </span>
                            </div>
                        </div>
                        <div class="card-body">
                            <div id="statusContainer">
                                <!-- Status will be loaded here -->
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- QR Code Connection -->
            <div class="row" id="connectionSection">
                <div class="col-md-8">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-qrcode mr-2"></i>Connect Your Phone
                            </h3>
                            <div class="card-tools">
                                <button type="button" class="btn btn-tool" onclick="refreshQRCode()">
                                    <i class="fas fa-refresh"></i>
                                </button>
                            </div>
                        </div>
                        <div class="card-body text-center">
                            <!-- QR Code Container -->
                            <div id="qrCodeContainer" class="mb-4">
                                <div class="loading-spinner"></div>
                                <p class="text-muted mt-2">Generating QR Code...</p>
                            </div>
                            
                            <!-- QR Code Timer -->
                            <div id="qrTimer" class="mb-3" style="display: none;">
                                <div class="progress" style="height: 8px;">
                                    <div class="progress-bar bg-success" role="progressbar" style="width: 100%" id="timerBar"></div>
                                </div>
                                <small class="text-muted">
                                    QR Code expires in <span id="timeLeft">30</span> seconds
                                </small>
                            </div>
                            
                            <!-- Action Buttons -->
                            <div class="btn-group">
                                <button type="button" class="btn btn-success" onclick="generateQRCode()" id="generateBtn">
                                    <i class="fas fa-qrcode mr-2"></i>Generate New QR
                                </button>
                                <button type="button" class="btn btn-info" onclick="openTestModal()" id="testBtn" disabled>
                                    <i class="fas fa-paper-plane mr-2"></i>Send Test Message
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-md-4">
                    <div class="card card-info">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-info-circle mr-2"></i>How to Connect
                            </h3>
                        </div>
                        <div class="card-body">
                            <div class="step-instructions">
                                <div class="step-item mb-3">
                                    <div class="step-number">1</div>
                                    <div class="step-content">
                                        <strong>Open WhatsApp</strong> on your phone
                                    </div>
                                </div>
                                <div class="step-item mb-3">
                                    <div class="step-number">2</div>
                                    <div class="step-content">
                                        Tap <strong>Menu</strong> (⋮) or go to <strong>Settings</strong>
                                    </div>
                                </div>
                                <div class="step-item mb-3">
                                    <div class="step-number">3</div>
                                    <div class="step-content">
                                        Select <strong>"Linked Devices"</strong>
                                    </div>
                                </div>
                                <div class="step-item mb-3">
                                    <div class="step-number">4</div>
                                    <div class="step-content">
                                        Tap <strong>"Link a Device"</strong>
                                    </div>
                                </div>
                                <div class="step-item">
                                    <div class="step-number">5</div>
                                    <div class="step-content">
                                        <strong>Scan this QR code</strong> with your camera
                                    </div>
                                </div>
                            </div>
                            
                            <div class="alert alert-warning mt-3">
                                <i class="fas fa-exclamation-triangle mr-2"></i>
                                <small>
                                    <strong>Note:</strong> Keep your phone connected to the internet. 
                                    Your phone acts as a bridge to send WhatsApp messages.
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Connected Device Info -->
            <div class="row" id="deviceSection" style="display: none;">
                <div class="col-12">
                    <div class="card card-success">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-mobile-alt mr-2"></i>Connected Device
                            </h3>
                            <div class="card-tools">
                                <button type="button" class="btn btn-tool text-white" onclick="disconnectDevice()">
                                    <i class="fas fa-unlink"></i> Disconnect
                                </button>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-3">
                                    <strong>Phone Number:</strong><br>
                                    <span id="devicePhone" class="text-success">N/A</span>
                                </div>
                                <div class="col-md-3">
                                    <strong>Device Name:</strong><br>
                                    <span id="deviceName" class="text-success">N/A</span>
                                </div>
                                <div class="col-md-3">
                                    <strong>Connected At:</strong><br>
                                    <span id="connectedAt" class="text-success">N/A</span>
                                </div>
                                <div class="col-md-3">
                                    <strong>Status:</strong><br>
                                    <span class="badge badge-success">
                                        <i class="fas fa-check-circle mr-1"></i>Active
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Test Message Modal -->
<div class="modal fade" id="testMessageModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-success">
                <h5 class="modal-title text-white">
                    <i class="fab fa-whatsapp mr-2"></i>Send Test WhatsApp Message
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="testMessageForm">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="testPhone">
                                    <i class="fas fa-phone mr-1"></i>Phone Number *
                                </label>
                                <input type="text" 
                                       id="testPhone" 
                                       class="form-control" 
                                       placeholder="e.g., +8801712345678"
                                       required>
                                <small class="text-muted">Include country code</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="messageType">
                                    <i class="fas fa-tag mr-1"></i>Message Type
                                </label>
                                <select id="messageType" class="form-control" onchange="updateMessageTemplate()">
                                    <option value="test">Test Message</option>
                                    <option value="invoice">Invoice Template</option>
                                    <option value="welcome">Welcome Message</option>
                                    <option value="custom">Custom Message</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="testMessage">
                            <i class="fas fa-comment mr-1"></i>Message *
                        </label>
                        <textarea id="testMessage" 
                                  class="form-control" 
                                  rows="5" 
                                  placeholder="Enter your test message here..."
                                  required></textarea>
                        <small class="text-muted">
                            You can use variables like {customer_name}, {amount}, {date}
                        </small>
                    </div>
                    
                    <div class="form-group">
                        <div class="custom-control custom-checkbox">
                            <input type="checkbox" class="custom-control-input" id="includeLink">
                            <label class="custom-control-label" for="includeLink">
                                Include website link in message
                            </label>
                        </div>
                    </div>
                </form>
                
                <!-- Message Preview -->
                <div class="mt-3">
                    <h6>Message Preview:</h6>
                    <div class="card bg-light">
                        <div class="card-body">
                            <div class="whatsapp-message">
                                <div class="message-bubble">
                                    <div id="messagePreview" class="message-text">
                                        Your message will appear here...
                                    </div>
                                    <div class="message-time text-muted">
                                        <small>{{ now()->format('h:i A') }}</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="fas fa-times mr-1"></i>Cancel
                </button>
                <button type="button" class="btn btn-success" onclick="sendTestMessage()">
                    <i class="fab fa-whatsapp mr-1"></i>Send Test Message
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('styles')
<style>
/* Loading Spinner */
.loading-spinner {
    border: 4px solid #f3f3f3;
    border-radius: 50%;
    border-top: 4px solid #25d366;
    width: 60px;
    height: 60px;
    animation: spin 1s linear infinite;
    margin: 20px auto;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* QR Code Styling */
.qr-code-frame {
    display: inline-block;
    padding: 25px;
    background: white;
    border-radius: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    border: 4px solid #25d366;
    position: relative;
}

.qr-code-frame::before {
    content: '';
    position: absolute;
    top: -8px;
    left: -8px;
    right: -8px;
    bottom: -8px;
    background: linear-gradient(45deg, #25d366, #128c7e);
    border-radius: 25px;
    z-index: -1;
}

.qr-code-frame img {
    border-radius: 12px;
    display: block;
    background: white;
    padding: 10px;
}

.qr-image {
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

/* WhatsApp-style message preview */
.whatsapp-preview {
    background: #e5ddd5;
    border-radius: 15px;
    padding: 20px;
    margin: 20px 0;
    position: relative;
}

.whatsapp-preview::before {
    content: '';
    position: absolute;
    top: 15px;
    left: 15px;
    width: 40px;
    height: 40px;
    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="%2325d366"><path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.886 3.488"/></svg>') no-repeat center;
    background-size: contain;
    opacity: 0.1;
}

/* Step Instructions Enhancement */
.step-instructions {
    position: relative;
    background: white;
    border-radius: 10px;
    padding: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.step-item {
    display: flex;
    align-items: flex-start;
    position: relative;
    margin-bottom: 15px;
    padding: 10px;
    border-radius: 8px;
    transition: background-color 0.2s;
}

.step-item:hover {
    background-color: #f8f9fa;
}

.step-number {
    flex: 0 0 35px;
    height: 35px;
    background: linear-gradient(135deg, #25d366, #128c7e);
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 16px;
    margin-right: 15px;
    position: relative;
    z-index: 2;
    box-shadow: 0 2px 8px rgba(37, 211, 102, 0.3);
}

.step-content {
    flex: 1;
    padding-top: 5px;
}

.step-item:not(:last-child)::after {
    content: '';
    position: absolute;
    left: 27px;
    top: 35px;
    width: 2px;
    height: calc(100% - 20px);
    background: linear-gradient(to bottom, #25d366, #ddd);
    z-index: 1;
}

/* Connection Status Indicators */
.badge-checking {
    background-color: #ffc107;
    animation: pulse-slow 2s infinite;
}

.badge-connected {
    background-color: #28a745;
    animation: none;
}

.badge-disconnected {
    background-color: #dc3545;
}

@keyframes pulse-slow {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.6; }
}

/* Alert Enhancements */
.alert {
    border: none;
    border-radius: 12px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.alert-info {
    background: linear-gradient(135deg, #17a2b8, #138496);
    color: white;
}

.alert-warning {
    background: linear-gradient(135deg, #ffc107, #e0a800);
    color: #212529;
}

.alert-success {
    background: linear-gradient(135deg, #28a745, #20c997);
    color: white;
}

.alert-danger {
    background: linear-gradient(135deg, #dc3545, #c82333);
    color: white;
}

/* QR Timer Styling */
#qrTimer {
    background: white;
    border-radius: 10px;
    padding: 10px 15px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    border: 2px solid #25d366;
}

#qrTimer .progress {
    height: 6px;
    border-radius: 3px;
    overflow: hidden;
}

/* Responsive Improvements */
@media (max-width: 768px) {
    .qr-code-frame {
        padding: 15px;
        margin: 0 auto;
    }
    
    .qr-code-frame img {
        max-width: 250px;
        height: auto;
    }
    
    .step-instructions {
        padding: 15px;
    }
    
    .step-number {
        width: 30px;
        height: 30px;
        font-size: 14px;
    }
    
    .step-item {
        padding: 8px;
        margin-bottom: 12px;
    }
    
    .connection-success-overlay .success-content {
        padding: 20px;
        margin: 0 20px;
        background: rgba(0,0,0,0.1);
        border-radius: 15px;
    }
}

/* Button Enhancements */
.btn {
    border-radius: 8px;
    font-weight: 500;
    transition: all 0.2s ease;
}

.btn-success {
    background: linear-gradient(135deg, #28a745, #20c997);
    border: none;
}

.btn-success:hover {
    background: linear-gradient(135deg, #218838, #1e9f85);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(40, 167, 69, 0.3);
}

.btn-info {
    background: linear-gradient(135deg, #17a2b8, #138496);
    border: none;
}

.btn-info:hover {
    background: linear-gradient(135deg, #138496, #0f6674);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(23, 162, 184, 0.3);
}

/* WhatsApp Message Preview */
.whatsapp-message {
    max-width: 300px;
    margin: 0 auto;
}

.message-bubble {
    background: #dcf8c6;
    padding: 8px 12px;
    border-radius: 18px;
    border-bottom-right-radius: 4px;
    position: relative;
    margin-left: auto;
    margin-right: 0;
    max-width: 250px;
}

.message-text {
    word-wrap: break-word;
    font-size: 14px;
    line-height: 1.4;
    margin-bottom: 4px;
}

.message-time {
    text-align: right;
    font-size: 11px;
}

/* Connection Status */
.badge-checking {
    background-color: #ffc107;
    animation: pulse 1.5s infinite;
}

.badge-connected {
    background-color: #28a745;
}

.badge-disconnected {
    background-color: #dc3545;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

/* Progress Bar Animation */
.progress-bar-animated {
    animation: progress-bar-stripes 1s linear infinite;
}

/* Card Enhancements */
.card {
    box-shadow: 0 0 15px rgba(0,0,0,0.1);
    border: none;
}

.card-outline.card-success {
    border-top: 3px solid #28a745;
}

/* Button Enhancements */
.btn-group .btn {
    margin: 0 5px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .step-instructions {
        font-size: 14px;
    }
    
    .step-number {
        width: 25px;
        height: 25px;
        font-size: 12px;
    }
    
    .qr-code-frame {
        padding: 10px;
    }
    
    .btn-group {
        flex-direction: column;
    }
    
    .btn-group .btn {
        margin: 5px 0;
        width: 100%;
    }
}
</style>
@endsection

@section('scripts')
<script>
let connectionCheckInterval;
let qrTimerInterval;
let currentQRSession = null;

$(document).ready(function() {
    initializeConnection();
});

function initializeConnection() {
    checkConnectionStatus();
    
    // Check connection status every 5 seconds
    connectionCheckInterval = setInterval(checkConnectionStatus, 5000);
    
    // Initialize message preview
    updateMessageTemplate();
    
    // Auto-update message preview as user types
    $('#testMessage').on('input', updateMessagePreview);
    $('#includeLink').on('change', updateMessagePreview);
}

function checkConnectionStatus() {
    $.ajax({
        url: '{{ route("whatsapp.check.connection") }}',
        method: 'GET',
        data: {
            simulate_connect: 'true'
        },
        success: function(response) {
            updateConnectionUI(response);
            
            // If just connected, show success animation
            if (response.just_connected) {
                showConnectionSuccess(response.device_info);
                
                // Clear polling intervals
                if (connectionCheckInterval) clearInterval(connectionCheckInterval);
                if (qrTimerInterval) clearInterval(qrTimerInterval);
            }
        },
        error: function() {
            updateConnectionUI({ connected: false, error: 'Connection check failed' });
        }
    });
}

function showConnectionSuccess(deviceInfo) {
    // Show success overlay
    const successOverlay = $(`
        <div class="connection-success-overlay" style="
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(40, 167, 69, 0.95);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            text-align: center;
        ">
            <div class="success-content">
                <div class="success-icon mb-4">
                    <i class="fas fa-check-circle" style="font-size: 80px; animation: bounce 1s;"></i>
                </div>
                <h2 class="mb-3">🎉 WhatsApp Connected!</h2>
                <p class="mb-2"><strong>Device:</strong> ${deviceInfo.device_name}</p>
                <p class="mb-4"><strong>Phone:</strong> ${deviceInfo.phone}</p>
                <button class="btn btn-light btn-lg" onclick="$('.connection-success-overlay').fadeOut()">
                    <i class="fas fa-arrow-right mr-2"></i>Continue to Dashboard
                </button>
            </div>
        </div>
    `);
    
    $('body').append(successOverlay);
    
    // Add bounce animation CSS if not exists
    if (!$('#bounce-animation').length) {
        $('head').append(`
            <style id="bounce-animation">
                @keyframes bounce {
                    0%, 20%, 53%, 80%, 100% {
                        transform: translate3d(0,0,0);
                    }
                    40%, 43% {
                        transform: translate3d(0,-30px,0);
                    }
                    70% {
                        transform: translate3d(0,-15px,0);
                    }
                    90% {
                        transform: translate3d(0,-4px,0);
                    }
                }
            </style>
        `);
    }
    
    // Auto-hide after 5 seconds
    setTimeout(function() {
        successOverlay.fadeOut(function() {
            $(this).remove();
        });
    }, 5000);
}

function simulateConnection() {
    // For testing - manually trigger connection
    if (confirm('Simulate WhatsApp QR scan and connection? (This is for testing the interface)')) {
        showAlert('Simulating QR code scan...', 'info');
        
        setTimeout(function() {
            $.ajax({
                url: '{{ route("whatsapp.check.connection") }}',
                method: 'GET',
                data: {
                    qr_scanned: 'true',
                    simulate_connect: 'true'
                },
                success: function(response) {
                    if (response.connected) {
                        showConnectionSuccess(response.device_info);
                        updateConnectionUI(response);
                        
                        // Clear intervals
                        if (connectionCheckInterval) clearInterval(connectionCheckInterval);
                        if (qrTimerInterval) clearInterval(qrTimerInterval);
                    }
                }
            });
        }, 2000);
    }
}

function updateConnectionUI(response) {
    const statusBadge = $('#connectionStatus');
    const statusContainer = $('#statusContainer');
    const connectionSection = $('#connectionSection');
    const deviceSection = $('#deviceSection');
    const testBtn = $('#testBtn');
    
    if (response.connected) {
        // Connected state
        statusBadge.removeClass('badge-secondary badge-checking badge-disconnected')
                   .addClass('badge-connected')
                   .html('<i class="fas fa-check-circle mr-1"></i>Connected');
        
        statusContainer.html(`
            <div class="alert alert-success mb-0">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h5 class="mb-1">
                            <i class="fas fa-check-circle mr-2"></i>WhatsApp Web Connected
                        </h5>
                        <p class="mb-0">Your WhatsApp is successfully linked and ready to send messages.</p>
                    </div>
                    <div class="col-md-4 text-right">
                        <div class="connection-indicators">
                            <span class="badge badge-success mr-2">
                                <i class="fas fa-wifi mr-1"></i>Online
                            </span>
                            <small class="text-muted">Last checked: ${new Date().toLocaleTimeString()}</small>
                        </div>
                    </div>
                </div>
            </div>
        `);
        
        // Show device info
        if (response.device_info) {
            $('#devicePhone').text(response.device_info.phone || 'N/A');
            $('#deviceName').text(response.device_info.device_name || 'N/A');
            $('#connectedAt').text(response.device_info.connected_at || 'Just now');
        }
        
        connectionSection.hide();
        deviceSection.show();
        testBtn.prop('disabled', false);
        
        // Clear QR timer if running
        if (qrTimerInterval) {
            clearInterval(qrTimerInterval);
        }
        
    } else {
        // Disconnected state
        statusBadge.removeClass('badge-secondary badge-checking badge-connected')
                   .addClass('badge-disconnected')
                   .html('<i class="fas fa-times-circle mr-1"></i>Disconnected');
        
        statusContainer.html(`
            <div class="alert alert-warning mb-0">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h5 class="mb-1">
                            <i class="fas fa-exclamation-triangle mr-2"></i>WhatsApp Not Connected
                        </h5>
                        <p class="mb-0">Connect your WhatsApp to start sending messages directly from the POS system.</p>
                    </div>
                    <div class="col-md-4 text-right">
                        <button class="btn btn-success btn-sm" onclick="generateQRCode()">
                            <i class="fas fa-qrcode mr-1"></i>Connect Now
                        </button>
                    </div>
                </div>
            </div>
        `);
        
        connectionSection.show();
        deviceSection.hide();
        testBtn.prop('disabled', true);
        
        // Generate QR code if not already generated
        if (!currentQRSession) {
            generateQRCode();
        }
    }
}

function generateQRCode() {
    const qrContainer = $('#qrCodeContainer');
    const generateBtn = $('#generateBtn');
    
    // Show loading state
    qrContainer.html(`
        <div class="loading-spinner"></div>
        <p class="text-muted mt-2">Generating WhatsApp Web QR code...</p>
    `);
    
    generateBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Generating...');
    
    // Generate WhatsApp Web compatible session data
    const timestamp = Date.now();
    const clientId = 'wa_web_' + timestamp + '_' + Math.random().toString(36).substr(2, 8);
    
    // Create WhatsApp Web format - this is the correct format that WhatsApp recognizes
    const whatsappWebData = {
        ref: btoa(clientId).replace(/=/g, ''),
        ttl: 30000, // 30 seconds
        type: 'web',
        timestamp: timestamp
    };
    
    setTimeout(function() {
        // WhatsApp Web uses a specific format: starts with "2@" followed by base64 encoded data
        const qrPayload = JSON.stringify({
            ref: whatsappWebData.ref,
            ttl: whatsappWebData.ttl,
            type: whatsappWebData.type,
            timestamp: whatsappWebData.timestamp,
            clientToken: clientId,
            serverToken: 'pos_server_' + timestamp
        });
        
        // This is the exact format WhatsApp Web uses - critical for device linking
        const whatsappQRData = `2@${btoa(qrPayload).replace(/=/g, '')}`;
        
        // Use multiple QR code services with proper error handling
        const qrServices = [
            // Google Charts - most reliable for WhatsApp format
            `https://chart.googleapis.com/chart?chs=300x300&cht=qr&chl=${encodeURIComponent(whatsappQRData)}&chld=L|0&choe=UTF-8`,
            // QR Server - backup service
            `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=${encodeURIComponent(whatsappQRData)}&ecc=L&format=png`,
            // QuickChart - third backup
            `https://quickchart.io/qr?text=${encodeURIComponent(whatsappQRData)}&size=300&margin=1&format=png`
        ];
        
        let currentServiceIndex = 0;
        
        function tryLoadQR() {
            // Test if the QR service is working
            const testImg = new Image();
            testImg.crossOrigin = "anonymous";
            
            testImg.onload = function() {
                // Successfully loaded QR code
                qrContainer.html(`
                    <div class="qr-code-frame">
                        <img src="${qrServices[currentServiceIndex]}" 
                             alt="WhatsApp Web QR Code" 
                             class="qr-image"
                             style="width: 300px; height: 300px; border-radius: 8px;">
                    </div>
                    <div class="qr-info mt-3">
                        <h5 class="text-success mb-2">
                            <i class="fab fa-whatsapp mr-2"></i>Ready to Scan
                        </h5>
                        <p class="mb-2">
                            <strong>Use WhatsApp's "Link a Device" feature</strong>
                        </p>
                        <div class="alert alert-info">
                            <small>
                                <i class="fas fa-info-circle mr-1"></i>
                                <strong>Important:</strong> Use WhatsApp's built-in QR scanner, not your regular camera app.
                            </small>
                        </div>
                    </div>
                    <div class="mt-3">
                        <button class="btn btn-outline-primary btn-sm mr-2" onclick="generateQRCode()">
                            <i class="fas fa-refresh mr-1"></i>Refresh QR
                        </button>
                        <button class="btn btn-outline-info btn-sm mr-2" onclick="showDetailedInstructions()">
                            <i class="fas fa-question-circle mr-1"></i>Need Help?
                        </button>
                        <button class="btn btn-outline-success btn-sm" onclick="simulateConnection()" title="For testing - simulate connection">
                            <i class="fas fa-magic mr-1"></i>Test Connect
                        </button>
                    </div>
                `);
                
                // Start the timer and connection polling
                startQRTimer();
                startConnectionPolling();
                
                // Store session for connection checking
                currentQRSession = {
                    id: clientId,
                    data: whatsappQRData,
                    timestamp: timestamp,
                    expires: timestamp + 30000
                };
            };
            
            testImg.onerror = function() {
                currentServiceIndex++;
                if (currentServiceIndex < qrServices.length) {
                    console.log(`QR service ${currentServiceIndex - 1} failed, trying service ${currentServiceIndex}`);
                    tryLoadQR(); // Try next service
                } else {
                    // All services failed
                    qrContainer.html(`
                        <div class="alert alert-danger text-center">
                            <i class="fas fa-exclamation-triangle fa-2x mb-3"></i>
                            <h5>QR Code Generation Failed</h5>
                            <p class="mb-3">Unable to generate WhatsApp QR code. This might be due to:</p>
                            <ul class="text-left">
                                <li>Internet connection issues</li>
                                <li>QR service provider problems</li>
                                <li>Firewall blocking external services</li>
                            </ul>
                            <button class="btn btn-primary mt-2" onclick="generateQRCode()">
                                <i class="fas fa-refresh mr-1"></i>Try Again
                            </button>
                            <button class="btn btn-info mt-2 ml-2" onclick="showManualInstructions()">
                                <i class="fas fa-book mr-1"></i>Manual Setup
                            </button>
                        </div>
                    `);
                }
            };
            
            // Load the image to test the service
            testImg.src = qrServices[currentServiceIndex];
        }
        
        tryLoadQR();
        
        generateBtn.prop('disabled', false).html('<i class="fas fa-qrcode mr-2"></i>Generate New QR');
        
    }, 1500);
}

function showDetailedInstructions() {
    const modal = $(`
        <div class="modal fade" id="instructionsModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header bg-success text-white">
                        <h5 class="modal-title">
                            <i class="fab fa-whatsapp mr-2"></i>How to Link Your WhatsApp Device
                        </h5>
                        <button type="button" class="close text-white" data-dismiss="modal">
                            <span>&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6 class="text-primary">📱 On Android:</h6>
                                <ol class="instructions-list">
                                    <li>Open <strong>WhatsApp</strong></li>
                                    <li>Tap the <strong>three dots (⋮)</strong> in top right</li>
                                    <li>Select <strong>"Linked devices"</strong></li>
                                    <li>Tap <strong>"Link a device"</strong></li>
                                    <li>Point camera at QR code</li>
                                </ol>
                            </div>
                            <div class="col-md-6">
                                <h6 class="text-primary">📱 On iPhone:</h6>
                                <ol class="instructions-list">
                                    <li>Open <strong>WhatsApp</strong></li>
                                    <li>Go to <strong>Settings</strong> tab</li>
                                    <li>Tap <strong>"Linked Devices"</strong></li>
                                    <li>Tap <strong>"Link a Device"</strong></li>
                                    <li>Point camera at QR code</li>
                                </ol>
                            </div>
                        </div>
                        
                        <div class="alert alert-warning mt-3">
                            <h6><i class="fas fa-exclamation-triangle mr-2"></i>Important Notes:</h6>
                            <ul class="mb-0">
                                <li><strong>Use WhatsApp's scanner:</strong> Don't use your phone's camera app</li>
                                <li><strong>Keep phone connected:</strong> Your phone needs internet to relay messages</li>
                                <li><strong>QR expires:</strong> Code refreshes every 30 seconds</li>
                                <li><strong>One device:</strong> You can only link one device at a time</li>
                            </ul>
                        </div>
                        
                        <div class="alert alert-info">
                            <h6><i class="fas fa-lightbulb mr-2"></i>Troubleshooting:</h6>
                            <ul class="mb-0">
                                <li>Make sure WhatsApp is updated to latest version</li>
                                <li>Check your internet connection</li>
                                <li>Try refreshing the QR code</li>
                                <li>Restart WhatsApp if scanning fails</li>
                            </ul>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                        <button type="button" class="btn btn-success" onclick="generateQRCode(); $('#instructionsModal').modal('hide');">
                            <i class="fas fa-refresh mr-1"></i>Refresh QR Code
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `);
    
    $('body').append(modal);
    modal.modal('show');
    
    // Remove modal when hidden
    modal.on('hidden.bs.modal', function() {
        $(this).remove();
    });
}

function showManualInstructions() {
    const modal = $(`
        <div class="modal fade" id="manualModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-info text-white">
                        <h5 class="modal-title">
                            <i class="fas fa-tools mr-2"></i>Manual Setup Option
                        </h5>
                        <button type="button" class="close text-white" data-dismiss="modal">
                            <span>&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <p>If QR code generation continues to fail, you can:</p>
                        
                        <div class="alert alert-primary">
                            <h6>Alternative Options:</h6>
                            <ol>
                                <li><strong>WhatsApp Business API:</strong> Set up official WhatsApp Business account</li>
                                <li><strong>Third-party Services:</strong> Use WhatsApp messaging services like Twilio</li>
                                <li><strong>Manual Messaging:</strong> Use WhatsApp web directly in browser</li>
                            </ol>
                        </div>
                        
                        <div class="form-group">
                            <label>Contact System Administrator:</label>
                            <textarea class="form-control" rows="3" readonly>
QR Code generation failed. Please check:
1. Internet connection
2. Firewall settings  
3. External API access
4. Server configuration

Session ID: ${currentQRSession?.id || 'N/A'}
Timestamp: ${new Date().toISOString()}
                            </textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                        <button type="button" class="btn btn-primary" onclick="generateQRCode(); $('#manualModal').modal('hide');">
                            <i class="fas fa-refresh mr-1"></i>Try QR Again
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `);
    
    $('body').append(modal);
    modal.modal('show');
    
    modal.on('hidden.bs.modal', function() {
        $(this).remove();
    });
}

function startQRTimer() {
    let timeLeft = 30;
    const timerDiv = $('#qrTimer');
    const timeLeftSpan = $('#timeLeft');
    const timerBar = $('#timerBar');
    
    timerDiv.show();
    
    if (qrTimerInterval) {
        clearInterval(qrTimerInterval);
    }
    
    qrTimerInterval = setInterval(function() {
        timeLeft--;
        timeLeftSpan.text(timeLeft);
        
        // Update progress bar
        const percentage = (timeLeft / 30) * 100;
        timerBar.css('width', percentage + '%');
        
        // Change color as time runs out
        if (timeLeft <= 10) {
            timerBar.removeClass('bg-success').addClass('bg-warning');
        }
        if (timeLeft <= 5) {
            timerBar.removeClass('bg-warning').addClass('bg-danger');
        }
        
        if (timeLeft <= 0) {
            clearInterval(qrTimerInterval);
            timerDiv.hide();
            
            // Auto-refresh QR code
            generateQRCode();
        }
    }, 1000);
}

function refreshQRCode() {
    if (qrTimerInterval) {
        clearInterval(qrTimerInterval);
    }
    $('#qrTimer').hide();
    generateQRCode();
}

function openTestModal() {
    $('#testMessageModal').modal('show');
    updateMessagePreview();
}

function updateMessageTemplate() {
    const messageType = $('#messageType').val();
    const messageTextarea = $('#testMessage');
    
    const templates = {
        test: 'Hello! This is a test message from our POS system. Your WhatsApp is successfully connected. 🎉',
        invoice: 'Hi {customer_name}! Your invoice #{invoice_id} for {amount} has been generated. Thank you for your business! 📄💰',
        welcome: 'Welcome to our store! We\'re excited to serve you. Feel free to contact us anytime for assistance. 😊',
        custom: ''
    };
    
    messageTextarea.val(templates[messageType] || '');
    updateMessagePreview();
}

function updateMessagePreview() {
    const message = $('#testMessage').val();
    const includeLink = $('#includeLink').is(':checked');
    
    let previewText = message || 'Your message will appear here...';
    
    // Replace common variables with sample data
    previewText = previewText
        .replace(/{customer_name}/g, 'John Doe')
        .replace(/{invoice_id}/g, '12345')
        .replace(/{amount}/g, '$150.00')
        .replace(/{date}/g, new Date().toLocaleDateString());
    
    if (includeLink && message) {
        previewText += '\\n\\nVisit our website: ' + window.location.origin;
    }
    
    $('#messagePreview').text(previewText);
}

function sendTestMessage() {
    const phone = $('#testPhone').val().trim();
    const message = $('#testMessage').val().trim();
    
    if (!phone) {
        showAlert('Please enter a phone number', 'warning');
        return;
    }
    
    if (!message) {
        showAlert('Please enter a message', 'warning');
        return;
    }
    
    const sendBtn = $('#testMessageModal .btn-success');
    const originalText = sendBtn.html();
    
    // Show loading state
    sendBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-1"></i>Sending...');
    
    // Include link if checked
    let finalMessage = message;
    if ($('#includeLink').is(':checked')) {
        finalMessage += '\\n\\nVisit our website: ' + window.location.origin;
    }
    
    $.ajax({
        url: '{{ route("whatsapp.send.test") }}',
        method: 'POST',
        data: {
            phone: phone,
            message: finalMessage,
            _token: '{{ csrf_token() }}'
        },
        success: function(response) {
            if (response.success) {
                showAlert('Test message sent successfully! 🎉', 'success');
                $('#testMessageModal').modal('hide');
                
                // Clear form
                $('#testPhone').val('');
                $('#testMessage').val('');
                $('#includeLink').prop('checked', false);
                updateMessageTemplate();
            } else {
                showAlert('Failed to send message: ' + (response.message || 'Unknown error'), 'danger');
            }
        },
        error: function(xhr) {
            let errorMessage = 'Failed to send test message';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                errorMessage = xhr.responseJSON.message;
            }
            showAlert(errorMessage, 'danger');
        },
        complete: function() {
            // Restore button
            sendBtn.prop('disabled', false).html(originalText);
        }
    });
}

function disconnectDevice() {
    if (!confirm('Are you sure you want to disconnect WhatsApp? You will need to scan the QR code again to reconnect.')) {
        return;
    }
    
    $.ajax({
        url: '{{ route("whatsapp.disconnect") }}',
        method: 'POST',
        data: {
            _token: '{{ csrf_token() }}'
        },
        success: function(response) {
            if (response.success) {
                showAlert('WhatsApp disconnected successfully', 'info');
                checkConnectionStatus();
            } else {
                showAlert('Failed to disconnect: ' + (response.message || 'Unknown error'), 'danger');
            }
        },
        error: function() {
            showAlert('Failed to disconnect WhatsApp. Please try again.', 'danger');
        }
    });
}

function showAlert(message, type = 'info') {
    const alertHtml = `
        <div class="alert alert-${type} alert-dismissible fade show" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
            ${message}
        </div>
    `;
    
    $('body').append(alertHtml);
    
    // Auto-hide after 5 seconds
    setTimeout(function() {
        $('.alert').alert('close');
    }, 5000);
}

// Cleanup on page unload
$(window).on('beforeunload', function() {
    if (connectionCheckInterval) clearInterval(connectionCheckInterval);
    if (qrTimerInterval) clearInterval(qrTimerInterval);
});
</script>
@endsection