<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use App\Models\Location;
use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $brands = Brand::all();
        return view('product.create', compact(['brands']));
    }

    /**
     * Store a newly created resource in storage.
     */

    public function store(Request $request)
    {
        try {
            // Validate the request data
            $validatedData = $request->validate([
                'customer_info' => 'nullable|integer',
                'brand' => 'required|integer',
                'model' => 'required|integer',
                'imei_or_sn' => 'required|string|max:255',
                'condition' => 'required|string|max:50',
                'color' => 'required|string|max:50',
                'gb' => 'required|integer',
                'buy_price' => 'required|numeric',
                'stock' => 'required|integer',
            ]);

            // Store the data in the database
            $product = Product::create([
                'customer_info' => $validatedData['customer_info'],
                'brand' => $validatedData['brand'],
                'model' => $validatedData['model'],
                'imei_or_sn' => $validatedData['imei_or_sn'],
                'condition' => $validatedData['condition'],
                'color' => $validatedData['color'],
                'gb' => $validatedData['gb'],
                'buy_price' => $validatedData['buy_price'],
                'stock' => $validatedData['stock'],
            ]);

            // Return a success response
            // return response()->json([
            //     'message' => 'Product added successfully',
            //     'product' => $product
            // ], 201); // HTTP 201 Created
            return redirect()->back()->with('success', $product->id);
        } catch (\Exception $e) {
            // Handle errors and exceptions
            return response()->json([
                'message' => 'An error occurred while adding the product',
                'error' => $e->getMessage()
            ], 500); // HTTP 500 Internal Server Error
        }
    }


    public function product_stock(Request $request)
    {
        $products = Product::with(['ProductBrand', 'ProductModel', 'Customer'])->get();
        // Calculate total cost (buy_price * stock)
        $totalCost = $products->sum(function ($product) {
            return $product->buy_price * $product->stock;
        });
        return view('product.stock', compact(['products', 'totalCost']));
    }

    public function fetchProducts(Request $request)
    {
        $searchType = $request->input('search_type', 'all');
        $searchValue = $request->input('search_value', '');

        $query = Product::with(['ProductBrand', 'ProductModel', 'Customer'])->where('stock', '>', 0);

        if ($searchType !== 'all' && !empty($searchValue)) {
            if ($searchType == 'model') {
                $query->whereHas('ProductModel', function ($q) use ($searchValue) {
                    $q->where('name', 'LIKE', "%$searchValue%");
                });
            } elseif ($searchType == 'brand') {
                $query->whereHas('ProductBrand', function ($q) use ($searchValue) {
                    $q->where('name', 'LIKE', "%$searchValue%");
                });
            } elseif ($searchType == 'imei') {
                $query->where('imei_or_sn', 'LIKE', "%$searchValue%");
            }
        }

        $products = $query->get();

        return response()->json($products);
    }



    public function print_receipt($id)
    {
        // return $id;
        $product = Product::with(['ProductBrand', 'ProductModel', 'Customer'])->findOrFail($id);
        $location = Location::first();
        return view('invoice.receipt', compact(['product', 'location']));
    }


    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        // return $products;
        return view('product.edit', compact(['product']));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        try {
            // Validate the request data
            $validatedData = $request->validate([
                'buy_price' => 'required|numeric',
                'stock_qty' => 'required|integer',
            ]);

            // Update the product in the database
            $product->update([
                'buy_price' => $validatedData['buy_price'],
                'stock' => $validatedData['stock_qty'],
            ]);

            // Redirect with success message
            return redirect()->route('product.stock')->with('success', "Product ID {$product->id} successfully updated!");
        } catch (\Exception $e) {
            // Redirect with error message
            return redirect()->back()->with('error', 'An error occurred while updating the product: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        $product->delete();
        return redirect()->back()->with('success', "Product successfully deleted!");
    }
}
