<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\DirectSales;
use App\Models\SellTransaction;
use App\Models\Expense;
use App\Models\Product;
use Barryvdh\DomPDF\Facade\Pdf;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\DB;

class AllReportController extends Controller
{
    public function index(Request $request)
    {
        // Get all direct sales and sell transactions
        $directSales = DirectSales::all();
        $sellTransactions = SellTransaction::all();

        // Apply filter if request is provided
        $filterDate = $request->input('filter_date', 'all');  // Default is 'all'

        if ($filterDate != 'all') {
            // Get the date range based on selected filter
            $dateRange = $this->getDateRange($filterDate);

            // Filter direct sales and sell transactions by the selected date range
            $directSales = DirectSales::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->get();
            $sellTransactions = SellTransaction::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->get();
        }

        // Calculate total profit from direct sales
        $directSalesProfit = $directSales->sum('direct_profit');

        // Calculate total profit from sell transactions
        $sellTransactionsProfit = $sellTransactions->sum('profit');

        // Calculate net profit
        $netProfit = $directSalesProfit + $sellTransactionsProfit;

        // Get the latest date from direct sales and sell transactions
        $latestDirectSaleDate = $directSales->max('created_at');
        $latestSellTransactionDate = $sellTransactions->max('created_at');

        // Determine the latest date overall
        $latestDate = max($latestDirectSaleDate, $latestSellTransactionDate);

        // Combine direct sales and sell transactions into one collection
        $combinedSales = collect($directSales->map(function ($sale) {
            return [
                'type' => 'Direct Sale',
                'date' => $sale->created_at,
                'profit' => $sale->direct_profit,
            ];
        }))->merge(collect($sellTransactions->map(function ($transaction) {
            return [
                'type' => 'Sell Transaction',
                'date' => $transaction->created_at,
                'profit' => $transaction->profit,
            ];
        })))->sortByDesc('date')->values();

        // Return view with the necessary data
        return view('report.index', compact('combinedSales', 'directSalesProfit', 'sellTransactionsProfit', 'netProfit', 'latestDate'));
    }

    public function profitLoss(Request $request)
    {
        $tenant = app('tenant');
        
        // Get settings for currency
        $all_settings = \App\Models\Setting::where('tenant_id', $tenant->id)->first();
        
        $filterDate = $request->input('filter_date', 'all');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // Get date range
        if ($filterDate === 'custom' && $startDate && $endDate) {
            $dateRange = [
                'start' => \Carbon\Carbon::parse($startDate)->startOfDay(),
                'end' => \Carbon\Carbon::parse($endDate)->endOfDay()
            ];
        } else {
            $dateRange = $this->getDateRange($filterDate);
        }

        // Revenue from Sell Transactions
        $sellTransactions = SellTransaction::where('tenant_id', $tenant->id)
            ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->get();

        $totalRevenueFromSales = $sellTransactions->sum(function ($transaction) {
            return $transaction->sell_price * $transaction->sold_quantity;
        });

        // Revenue from Direct Sales
        $directSales = DirectSales::where('tenant_id', $tenant->id)
            ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->get();

        $totalRevenueFromDirectSales = $directSales->sum('direct_sell_price');

        // Total Revenue
        $totalRevenue = $totalRevenueFromSales + $totalRevenueFromDirectSales;

        // Cost of Goods Sold (COGS) - based on buy_price
        $cogsFromSales = $sellTransactions->sum(function ($transaction) {
            return $transaction->product->buy_price * $transaction->sold_quantity;
        });

        $cogsFromDirectSales = $directSales->sum('direct_cost_price');

        $totalCOGS = $cogsFromSales + $cogsFromDirectSales;

        // Gross Profit
        $grossProfit = $totalRevenue - $totalCOGS;

        // Operating Expenses
        $expenses = Expense::where('tenant_id', $tenant->id)
            ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->get();

        $totalExpenses = $expenses->sum('amount');

        // Net Profit
        $netProfit = $grossProfit - $totalExpenses;

        // Profit from Sales
        $profitFromSales = $sellTransactions->sum('profit');
        $profitFromDirectSales = $directSales->sum('direct_profit');
        $totalProfit = $profitFromSales + $profitFromDirectSales;

        // Data for charts - Monthly profit data for the last 12 months
        $monthlyProfits = [];
        for ($i = 11; $i >= 0; $i--) {
            $monthStart = \Carbon\Carbon::now()->subMonths($i)->startOfMonth();
            $monthEnd = \Carbon\Carbon::now()->subMonths($i)->endOfMonth();

            $monthSalesProfit = SellTransaction::where('tenant_id', $tenant->id)
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->sum('profit');

            $monthDirectProfit = DirectSales::where('tenant_id', $tenant->id)
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->sum('direct_profit');

            $monthlyProfits[] = [
                'month' => $monthStart->format('M Y'),
                'profit' => $monthSalesProfit + $monthDirectProfit
            ];
        }

        // Top selling products
        $topProducts = SellTransaction::select('product_id', DB::raw('SUM(sold_quantity) as total_sold'), DB::raw('SUM(profit) as total_profit'))
            ->where('tenant_id', $tenant->id)
            ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->groupBy('product_id')
            ->with('product')
            ->orderBy('total_sold', 'desc')
            ->take(10)
            ->get();

        $data = [
            'totalRevenue' => $totalRevenue,
            'totalCOGS' => $totalCOGS,
            'grossProfit' => $grossProfit,
            'totalExpenses' => $totalExpenses,
            'netProfit' => $netProfit,
            'totalProfit' => $totalProfit,
            'profitFromSales' => $profitFromSales,
            'profitFromDirectSales' => $profitFromDirectSales,
            'monthlyProfits' => $monthlyProfits,
            'topProducts' => $topProducts,
            'filterDate' => $filterDate,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'dateRange' => $dateRange
        ];

        if ($request->has('export') && $request->export === 'pdf') {
            return $this->exportPDF($data);
        }

        if ($request->has('export') && $request->export === 'excel') {
            return $this->exportExcel($data);
        }

        return view('profit-report', compact('data', 'all_settings'));

    private function exportPDF($data)
    {
        $tenant = app('tenant');
        $all_settings = \App\Models\Setting::where('tenant_id', $tenant->id)->first();
        
        $pdf = Pdf::loadView('reports.profit-loss-pdf', compact('data', 'all_settings'));
        return $pdf->download('profit-loss-report.pdf');
    }

    private function exportExcel($data)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set headers
        $sheet->setCellValue('A1', 'Profit & Loss Report');
        $sheet->setCellValue('A3', 'Period: ' . ($data['filterDate'] === 'all' ? 'All Time' : $data['filterDate']));

        // Financial Summary
        $sheet->setCellValue('A5', 'Financial Summary');
        $sheet->setCellValue('A6', 'Total Revenue');
        $sheet->setCellValue('B6', $data['totalRevenue']);
        $sheet->setCellValue('A7', 'Cost of Goods Sold');
        $sheet->setCellValue('B7', $data['totalCOGS']);
        $sheet->setCellValue('A8', 'Gross Profit');
        $sheet->setCellValue('B8', $data['grossProfit']);
        $sheet->setCellValue('A9', 'Total Expenses');
        $sheet->setCellValue('B9', $data['totalExpenses']);
        $sheet->setCellValue('A10', 'Net Profit');
        $sheet->setCellValue('B10', $data['netProfit']);

        // Top Products
        $sheet->setCellValue('A12', 'Top Selling Products');
        $sheet->setCellValue('A13', 'Product');
        $sheet->setCellValue('B13', 'Total Sold');
        $sheet->setCellValue('C13', 'Total Profit');

        $row = 14;
        foreach ($data['topProducts'] as $product) {
            $sheet->setCellValue('A' . $row, $product->product->name ?? 'N/A');
            $sheet->setCellValue('B' . $row, $product->total_sold);
            $sheet->setCellValue('C' . $row, $product->total_profit);
            $row++;
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'profit-loss-report.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    private function getDateRange($filterDate)
    {
        switch ($filterDate) {
            case 'today':
                $start = \Carbon\Carbon::today()->startOfDay();
                $end = \Carbon\Carbon::today()->endOfDay();
                break;
            case 'yesterday':
                $start = \Carbon\Carbon::yesterday()->startOfDay();
                $end = \Carbon\Carbon::yesterday()->endOfDay();
                break;
            case 'last_7_days':
                $start = \Carbon\Carbon::now()->subDays(7)->startOfDay();
                $end = \Carbon\Carbon::now()->endOfDay();
                break;
            case 'last_30_days':
                $start = \Carbon\Carbon::now()->subDays(30)->startOfDay();
                $end = \Carbon\Carbon::now()->endOfDay();
                break;
            case 'last_year':
                $start = \Carbon\Carbon::now()->subYear()->startOfYear();
                $end = \Carbon\Carbon::now()->subYear()->endOfYear();
                break;
            default:
                // Use a very old date as the start and a very future date as the end
                $start = \Carbon\Carbon::createFromFormat('Y-m-d H:i:s', '2000-01-01 00:00:00');
                $end = \Carbon\Carbon::createFromFormat('Y-m-d H:i:s', '2100-01-01 23:59:59');
                break;
        }

        return ['start' => $start, 'end' => $end];
    }
}
