<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerPayment;
use App\Models\Sale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $tenant = app('tenant');
        
        if ($request->ajax()) {
            $query = Customer::with(['sales' => function ($query) {
                $query->select('id', 'customer_id', 'created_at', 'total_amount', 'paid_amount', 'due_amount', 'payment_status', 'payment_method')
                    ->where('return_status', '!=', 'returned')
                    ->orderBy('created_at', 'desc');
            }])->where('tenant_id', $tenant->id);

            // Apply search filter
            if ($request->has('search') && !empty($request->search)) {
                $searchTerm = $request->search;
                $query->where(function($q) use ($searchTerm) {
                    $q->where('name', 'LIKE', "%{$searchTerm}%")
                      ->orWhere('email', 'LIKE', "%{$searchTerm}%")
                      ->orWhere('phone', 'LIKE', "%{$searchTerm}%")
                      ->orWhere('address', 'LIKE', "%{$searchTerm}%");
                });
            }

            // Apply date filter
            if ($request->has('date_from') && !empty($request->date_from)) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }
            if ($request->has('date_to') && !empty($request->date_to)) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            // Apply due amount filter
            $dueFilter = $request->get('due_filter');
            
            $totalRecords = Customer::where('tenant_id', $tenant->id)->count();
            $customers = $query->get()->map(function ($customer) {
            $totalDueFromSales = $customer->sales->sum('due_amount');
            $advances = CustomerPayment::where('customer_id', $customer->id)
                ->where('type', 'advance')
                ->sum('amount');
            // Only subtract advances since due_amount in sales already reflects payments made
            $netDue = $totalDueFromSales - $advances;
            $totalDue = max(0, $netDue);
            $advance = max(0, -$netDue);

            $totalPaid = $customer->sales->sum('paid_amount');
            $totalCredit = $customer->sales->sum('total_amount');

            $customer->exchange_amount = $customer->sales->sum(function ($sale) {
                return $sale->exchanges->sum('price');
            });

            $customer->total_due = $totalDue;
            $customer->total_paid = $totalPaid;
            $customer->total_credit = $totalCredit;
            $customer->advance = $advance;

            // Count sales by type
            $customer->cash_sales = $customer->sales->where('payment_method', 'cash')->count();
            $customer->credit_sales = $customer->sales->where('payment_method', 'credit')->count();
            $customer->exchange_sales = $customer->sales->filter(function ($sale) {
                return $sale->exchanges->count() > 0;
            })->count();

            return $customer;
        });

            // Apply due filter after calculations
            if ($dueFilter) {
                $customers = $customers->filter(function($customer) use ($dueFilter) {
                    switch($dueFilter) {
                        case 'has_due':
                            return $customer->total_due > 0;
                        case 'no_due':
                            return $customer->total_due <= 0;
                        case 'high_due':
                            return $customer->total_due > 1000;
                        default:
                            return true;
                    }
                });
            }

            $filteredRecords = $customers->count();

            $data = [];
            foreach ($customers as $index => $customer) {
                $dueBadge = '';
                if ($customer->total_due > 1000) {
                    $dueBadge = '<span class="badge badge-danger">High Due</span>';
                } elseif ($customer->total_due > 0) {
                    $dueBadge = '<span class="badge badge-warning">Has Due</span>';
                } else {
                    $dueBadge = '<span class="badge badge-success">No Due</span>';
                }

                $data[] = [
                    'id' => $customer->id,
                    'sl' => $index + 1,
                    'name' => '<strong>' . $customer->name . '</strong>',
                    'email' => $customer->email ?: '<em class="text-muted">No Email</em>',
                    'mobile' => $customer->phone ?: '<em class="text-muted">No Mobile</em>',
                    'address' => $customer->address ? (strlen($customer->address) > 30 ? substr($customer->address, 0, 30) . '...' : $customer->address) : '<em class="text-muted">No Address</em>',
                    'total_due' => '<span class="text-danger">BHD ' . number_format($customer->total_due, 2) . '</span> ' . $dueBadge,
                    'total_paid' => '<span class="text-success">BHD ' . number_format($customer->total_paid, 2) . '</span>',
                    'total_credit' => '<span class="text-info">BHD ' . number_format($customer->total_credit, 2) . '</span>',
                    'advance' => $customer->advance > 0 ? '<span class="text-primary">BHD ' . number_format($customer->advance, 2) . '</span>' : '<em class="text-muted">No Advance</em>',
                    'cash_sales' => '<span class="badge badge-success">' . $customer->cash_sales . '</span>',
                    'credit_sales' => '<span class="badge badge-warning">' . $customer->credit_sales . '</span>',
                    'exchange_sales' => '<span class="badge badge-info">' . $customer->exchange_sales . '</span>',
                    'created_at' => $customer->created_at->format('d M, Y'),
                    'actions' => view('customers.partials.actions', compact('customer'))->render(),
                ];
            }

            return response()->json([
                'data' => $data,
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords
            ]);
        }

        $customers = Customer::with(['sales' => function ($query) {
            $query->select('id', 'customer_id', 'created_at', 'total_amount', 'paid_amount', 'due_amount', 'payment_status', 'payment_method')
                ->where('return_status', '!=', 'returned')
                ->orderBy('created_at', 'desc');
        }])->where('tenant_id', $tenant->id)->get()->map(function ($customer) {
            $totalDueFromSales = $customer->sales->sum('due_amount');
            $advances = CustomerPayment::where('customer_id', $customer->id)
                ->where('type', 'advance')
                ->sum('amount');
            $netDue = $totalDueFromSales - $advances;
            $totalDue = max(0, $netDue);
            $advance = max(0, -$netDue);
            $totalPaid = $customer->sales->sum('paid_amount');
            $totalCredit = $customer->sales->sum('total_amount');

            $customer->total_due = $totalDue;
            $customer->total_paid = $totalPaid;
            $customer->total_credit = $totalCredit;
            $customer->advance = $advance;

            return $customer;
        });

        return view('customers.index', compact('customers'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('customers.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Handle both API requests (customer_name) and form submissions (name)
            if ($request->has('customer_name')) {
                // API request validation
                $validatedData = $request->validate([
                    'customer_name'   => 'required|string|max:255',
                    'customer_number' => 'required|string|max:20|unique:customers,phone',
                    'customer_cpr'    => 'nullable|string|max:20',
                    'address'         => 'nullable|string|max:255',
                    'email'           => 'nullable|email|max:255',
                    'cpr_photo'       => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                ]);

                $customerData = [
                    'name'    => $validatedData['customer_name'],
                    'phone'   => $validatedData['customer_number'],
                    'cpr'     => $validatedData['customer_cpr'] ?? null,
                    'address' => $validatedData['address'] ?? null,
                    'email'   => $validatedData['email'] ?? null,
                ];
            } else {
                // Form submission validation
                $validatedData = $request->validate([
                    'name'    => 'required|string|max:255',
                    'phone'   => 'required|string|max:20|unique:customers,phone',
                    'cpr'     => 'nullable|string|max:20',
                    'address' => 'nullable|string|max:255',
                    'email'   => 'nullable|email|max:255',
                    'cpr_photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                ]);

                $customerData = $validatedData;
            }

            $tenant = app('tenant');

            // Handle CPR photo upload if present
            $photoPath = null;
            $photoSaved = false;
            if ($request->hasFile('cpr_photo')) {
                try {
                    // Create customer-photos directory if it doesn't exist
                    $photoDirectory = 'customer-photos';
                    if (!file_exists(public_path($photoDirectory))) {
                        mkdir(public_path($photoDirectory), 0755, true);
                    }
                    
                    $file = $request->file('cpr_photo');
                    $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                    $file->move(public_path($photoDirectory), $filename);
                    $photoPath = $photoDirectory . '/' . $filename;
                    $photoSaved = true;
                } catch (\Exception $e) {
                    // Log error but don't fail the customer creation
                    \Log::error('Failed to save customer photo: ' . $e->getMessage());
                }
            }

            // Insert customer into the database
            $customer = Customer::create([
                'tenant_id' => $tenant->id,
                'name'    => $customerData['name'],
                'phone'   => $customerData['phone'],
                'cpr'     => $customerData['cpr'] ?? null,
                'address' => $customerData['address'] ?? null,
                'email'   => $customerData['email'] ?? null,
                'cpr_photo' => $photoPath,
            ]);

            // Check if it's an API request or form submission
            if ($request->expectsJson() || $request->has('customer_name')) {
                // API response
                $customerInfo = [
                    'id'     => $customer->id,
                    'name'   => $customer->name,
                    'number' => $customer->phone,
                    'cpr'    => $customer->cpr,
                    'email'  => $customer->email,
                    'cpr_photo' => $customer->cpr_photo,
                ];

                return response()->json([
                    'message'  => 'Customer added successfully',
                    'customer' => $customerInfo,
                    'photo_saved' => $photoSaved
                ], 201);
            } else {
                // Form submission response
                $message = 'Customer added successfully' . ($photoSaved ? ' with photo!' : '.');
                return redirect()->route('customer.index')->with('success', $message);
            }

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson() || $request->has('customer_name')) {
                return response()->json([
                    'message' => 'Validation failed',
                    'errors'  => $e->errors(),
                ], 422);
            } else {
                return redirect()->back()->withErrors($e->errors())->withInput();
            }

        } catch (\Illuminate\Database\UniqueConstraintViolationException $e) {
            if ($request->expectsJson() || $request->has('customer_name')) {
                return response()->json([
                    'message' => 'This phone number is already in use by another customer.',
                    'error'   => 'duplicate_phone',
                ], 422);
            } else {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'This phone number is already in use by another customer.');
            }

        } catch (\Exception $e) {
            if ($request->expectsJson() || $request->has('customer_name')) {
                return response()->json([
                    'message' => 'An error occurred while adding the customer',
                    'error'   => $e->getMessage(),
                ], 500);
            } else {
                return redirect()->back()->with('error', 'An error occurred while adding the customer.')->withInput();
            }
        }
    }
    /**
     * Display the specified resource.
     */
    public function show(Customer $customer)
    {
        $tenant = app('tenant');
        $customer = Customer::where('tenant_id', $tenant->id)->findOrFail($customer->id);

        $customer->load(['sales' => function ($query) {
            $query->with('payments')->orderBy('created_at', 'desc');
        }]);

        $payments = CustomerPayment::where('customer_id', $customer->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $totalDueFromSales = $customer->sales->sum('due_amount');
        $advances = $payments->where('type', 'advance')->sum('amount');
        // Only subtract advances since due_amount in sales already reflects payments made
        $netDue = $totalDueFromSales - $advances;
        $totalDue = max(0, $netDue);
        $advance = max(0, -$netDue);

        $totalPaid = $customer->sales->sum('paid_amount');
        $totalCredit = $customer->sales->sum('total_amount');

        return view('customers.show', compact('customer', 'payments', 'totalDue', 'totalPaid', 'totalCredit', 'advance'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Customer $customer)
    {
        return view('customers.edit', compact('customer'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Customer $customer)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20|unique:customers,phone,' . $customer->id,
            'email' => 'nullable|email|max:255',
            'cpr' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
        ]);

        try {
            $customer->update([
                'name' => $request->name,
                'phone' => $request->phone,
                'email' => $request->email,
                'cpr' => $request->cpr,
                'address' => $request->address,
            ]);

            return redirect()->route('customer.index')->with('success', 'Customer updated successfully.');
            
        } catch (\Illuminate\Database\UniqueConstraintViolationException $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'This phone number is already in use by another customer.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'An error occurred while updating the customer: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Customer $customer)
    {
        try {
            // Check if customer has any sales or payments
            $hasSales = $customer->sales()->count() > 0;
            $hasPayments = CustomerPayment::where('customer_id', $customer->id)->count() > 0;
            
            if ($hasSales || $hasPayments) {
                return response()->json([
                    'message' => 'Cannot delete customer with existing sales or payment records.',
                ], 400);
            }
            
            $customer->delete();
            
            return response()->json([
                'message' => 'Customer deleted successfully.',
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An error occurred while deleting the customer.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function checkPhone(Request $request)
    {
        $phone = $request->input('phone');
        $tenant = app('tenant');
        $customer = Customer::where('tenant_id', $tenant->id)->where('phone', $phone)->first();

        if ($customer) {
            return response()->json([
                'exists' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'phone' => $customer->phone,
                    'cpr' => $customer->cpr,
                    'address' => $customer->address,
                    'email' => $customer->email,
                ]
            ]);
        } else {
            return response()->json(['exists' => false]);
        }
    }

    public function list()
    {
        $tenant = app('tenant');
        $customers = Customer::where('tenant_id', $tenant->id)->select('id', 'name', 'phone')->get();
        return response()->json($customers);
    }


    public function credits()
    {
        $tenant = app('tenant');
        $customers = Customer::with(['sales' => function ($query) {
            $query->select('id', 'customer_id', 'created_at', 'total_amount', 'due_amount', 'payment_status')
                ->where('due_amount', '!=', 0)
                ->orderBy('created_at', 'desc');
        }])->where('tenant_id', $tenant->id)->get()->map(function ($customer) {
            $totalDueFromSales = $customer->sales->sum('due_amount');
            $duePayments = CustomerPayment::where('customer_id', $customer->id)
                ->where('type', 'due_payment')
                ->sum('amount');
            $advances = CustomerPayment::where('customer_id', $customer->id)
                ->where('type', 'advance')
                ->sum('amount');
        $netDue = $totalDueFromSales - $duePayments - $advances;
        $totalDue = max(0, $netDue);
        $advance = max(0, -$netDue);

        $customer->total_due = $totalDue;
        $customer->advance = $advance;
            return $customer;
        });

        return view('customers.credits', compact('customers'));
    }

    public function search(Request $request)
    {
        $query = $request->input('query');
        $tenant = app('tenant');

        if (!$query) {
            $customers = Customer::where('tenant_id', $tenant->id)->select('id', 'name', 'phone')->get();
        } else {
            // Search by name or phone
            $customers = Customer::where('tenant_id', $tenant->id)
                ->where(function ($q) use ($query) {
                    $q->where('name', 'LIKE', "%{$query}%")
                      ->orWhere('phone', 'LIKE', "%{$query}%");
                })
                ->take(10)
                ->get(['id', 'name', 'phone']);
        }

        return response()->json($customers);
    }

    public function payDue(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'amount' => 'required|numeric|min:0.01',
            'type' => 'required|in:due_payment,advance',
            'notes' => 'nullable|string',
        ]);

        $customer = Customer::findOrFail($request->customer_id);
        $amount = $request->amount;

        $tenant = app('tenant');

        if ($request->type === 'due_payment') {
            // Handle payment towards previous due
            $unpaidSales = Sale::where('tenant_id', $tenant->id)->where('customer_id', $customer->id)
                ->where('due_amount', '>', 0)
                ->get();

            $totalDue = $unpaidSales->sum('due_amount');

            if ($amount <= $totalDue) {
                // Distribute sequentially from oldest
                $paidSaleIds = [];
                $remainingPay = $amount;
                $unpaidSalesOrdered = Sale::where('tenant_id', $tenant->id)->where('customer_id', $customer->id)
                    ->where('due_amount', '>', 0)
                    ->orderBy('created_at', 'asc')
                    ->get();

                foreach ($unpaidSalesOrdered as $sale) {
                    if ($remainingPay <= 0) break;
                    $payAmount = min($remainingPay, $sale->due_amount);
                    $sale->paid_amount += $payAmount;
                    $sale->due_amount -= $payAmount;
                    if ($sale->due_amount <= 0) {
                        $sale->payment_status = 'paid';
                    } else {
                        $sale->payment_status = 'partial';
                    }
                    $sale->save();
                    $paidSaleIds[] = $sale->id;
                    $remainingPay -= $payAmount;
                }

                if (!empty($paidSaleIds)) {
                    CustomerPayment::create([
                        'tenant_id' => $tenant->id,
                        'customer_id' => $customer->id,
                        'type' => 'due_payment',
                        'amount' => $amount - $remainingPay,
                        'sale_ids' => json_encode($paidSaleIds),
                        'notes' => $request->notes ?: 'Due payment for sales: ' . implode(', ', $paidSaleIds),
                    ]);
                }
            } else {
                // Pay full to all, advance the rest
                $paidSaleIds = [];
                $paidAmount = 0;
                foreach ($unpaidSales as $sale) {
                    $payAmount = $sale->due_amount;
                    $sale->paid_amount += $payAmount;
                    $sale->due_amount = 0;
                    $sale->payment_status = 'paid';
                    $sale->save();
                    $paidSaleIds[] = $sale->id;
                    $paidAmount += $payAmount;
                }

                if (!empty($paidSaleIds)) {
                    CustomerPayment::create([
                        'tenant_id' => $tenant->id,
                        'customer_id' => $customer->id,
                        'type' => 'due_payment',
                        'amount' => $paidAmount,
                        'sale_ids' => json_encode($paidSaleIds),
                        'notes' => $request->notes ?: 'Due payment for sales: ' . implode(', ', $paidSaleIds),
                    ]);
                }

                $advanceAmount = $amount - $paidAmount;
                CustomerPayment::create([
                    'tenant_id' => $tenant->id,
                    'customer_id' => $customer->id,
                    'type' => 'advance',
                    'amount' => $advanceAmount,
                    'sale_ids' => null,
                    'notes' => 'Advance payment',
                ]);
            }

            $message = 'Due payment processed successfully';
        } else {
            // Advance payment
            $customerPayment = CustomerPayment::create([
                'tenant_id' => $tenant->id,
                'customer_id' => $customer->id,
                'type' => 'advance',
                'amount' => $amount,
                'sale_ids' => null,
                'notes' => $request->notes ?: 'Advance payment',
            ]);
            $message = 'Advance payment recorded successfully';
        }

        return response()->json([
            'message' => $message,
        ]);
    }
}
