<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class InvoiceSettingsController extends Controller
{
    /**
     * Display the invoice settings page.
     */
    public function index()
    {
        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        // Default settings if none exist
        if (!$settings) {
            $settings = new Setting([
                'tenant_id' => $tenant->id,
                'company_name' => 'Your Company Name',
                'currency' => 'BDT',
                'whatsapp_enabled' => true,
                'invoice_template' => 'default',
                'invoice_message_template' => $this->getDefaultMessageTemplate(),
                'invoice_terms_conditions' => $this->getDefaultTermsConditions(),
                'invoice_header_text' => 'INVOICE',
                'invoice_footer_text' => 'Thank you for your business!',
                'auto_send_whatsapp' => false,
                'include_company_logo' => true,
                'include_qr_code' => false,
                'whatsapp_country_code' => '+880',
                'whatsapp_welcome_message' => $this->getDefaultWelcomeMessage(),
                'whatsapp_payment_reminder_message' => $this->getDefaultPaymentReminderMessage(),
                'whatsapp_delivery_message' => $this->getDefaultDeliveryMessage(),
                'whatsapp_thank_you_message' => $this->getDefaultThankYouMessage(),
                'whatsapp_auto_format_numbers' => true,
                'whatsapp_retry_attempts' => 3,
                'whatsapp_include_business_hours' => false,
                'whatsapp_business_hours' => '9:00 AM - 6:00 PM'
            ]);
        }
        
        return view('invoice-settings.index', compact('settings'));
    }

    /**
     * Update the invoice settings.
     */
    public function update(Request $request)
    {
        $request->validate([
            'company_name' => 'required|string|max:255',
            'invoice_header_text' => 'required|string|max:100',
            'invoice_footer_text' => 'nullable|string|max:500',
            'invoice_terms_conditions' => 'nullable|string|max:2000',
            'whatsapp_enabled' => 'boolean',
            'auto_send_whatsapp' => 'boolean',
            'include_company_logo' => 'boolean',
            'include_qr_code' => 'boolean',
            'invoice_template' => 'required|in:default,modern,classic,minimal',
            'whatsapp_country_code' => 'required|string|max:10',
            'whatsapp_welcome_message' => 'nullable|string|max:1000',
            'whatsapp_payment_reminder_message' => 'nullable|string|max:1000',
            'whatsapp_delivery_message' => 'nullable|string|max:1000',
            'whatsapp_thank_you_message' => 'nullable|string|max:1000',
            'whatsapp_invoice_message_template' => 'nullable|string|max:2000',
            'whatsapp_auto_format_numbers' => 'boolean',
            'whatsapp_include_business_hours' => 'boolean',
            'whatsapp_business_hours' => 'nullable|string|max:255',
        ]);

        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        if (!$settings) {
            $settings = new Setting(['tenant_id' => $tenant->id]);
        }

        $settings->fill([
            'company_name' => $request->company_name,
            'invoice_header_text' => $request->invoice_header_text,
            'invoice_footer_text' => $request->invoice_footer_text,
            'invoice_terms_conditions' => $request->invoice_terms_conditions,
            'whatsapp_enabled' => $request->boolean('whatsapp_enabled'),
            'auto_send_whatsapp' => $request->boolean('auto_send_whatsapp'),
            'include_company_logo' => $request->boolean('include_company_logo'),
            'include_qr_code' => $request->boolean('include_qr_code'),
            'invoice_template' => $request->invoice_template,
            'whatsapp_country_code' => $request->whatsapp_country_code,
            'whatsapp_welcome_message' => $request->whatsapp_welcome_message,
            'whatsapp_payment_reminder_message' => $request->whatsapp_payment_reminder_message,
            'whatsapp_delivery_message' => $request->whatsapp_delivery_message,
            'whatsapp_thank_you_message' => $request->whatsapp_thank_you_message,
            'whatsapp_invoice_message_template' => $request->whatsapp_invoice_message_template ?: $this->getDefaultMessageTemplate(),
            'whatsapp_auto_format_numbers' => $request->boolean('whatsapp_auto_format_numbers'),
            'whatsapp_include_business_hours' => $request->boolean('whatsapp_include_business_hours'),
            'whatsapp_business_hours' => $request->whatsapp_business_hours,
        ]);

        $settings->save();

        return redirect()->route('invoice-settings.index')
            ->with('success', 'Invoice settings updated successfully!');
    }

    /**
     * Preview the invoice template.
     */
    public function preview(Request $request)
    {
        $template = $request->get('template', 'default');
        $settings = Setting::where('tenant_id', app('tenant')->id)->first();
        
        // Sample data for preview
        $sampleData = [
            'invoice_id' => 'INV-2025-001',
            'customer_name' => 'John Doe',
            'customer_phone' => '+8801712345678',
            'date' => now()->format('Y-m-d'),
            'items' => [
                ['name' => 'iPhone 15 Pro', 'qty' => 1, 'price' => 120000, 'total' => 120000],
                ['name' => 'Samsung Galaxy S24', 'qty' => 2, 'price' => 95000, 'total' => 190000]
            ],
            'subtotal' => 310000,
            'discount' => 10000,
            'total' => 300000
        ];
        
        return view('invoice-settings.preview', compact('template', 'settings', 'sampleData'));
    }

    /**
     * Get default WhatsApp message template.
     */
    private function getDefaultMessageTemplate()
    {
        return "🧾 *Invoice #{invoice_number}*\n\n" .
               "👤 *Customer:* {customer_name}\n" .
               "📞 *Phone:* {customer_phone}\n" .
               "📅 *Date:* {invoice_date}\n\n" .
               "📦 *Items:*\n{item_list}\n\n" .
               "💰 *Total Amount:* {currency} {total_amount}\n" .
               "� *Paid Amount:* {currency} {paid_amount}\n" .
               "⚠️ *Due Amount:* {currency} {due_amount}\n" .
               "�💳 *Payment Method:* {payment_method}\n" .
               "📊 *Payment Status:* {payment_status}\n\n" .
               "🙏 *Thank you for your business!*\n\n" .
               "🔗 *View Invoice:* {invoice_link}";
    }

    /**
     * Get default terms and conditions.
     */
    private function getDefaultTermsConditions()
    {
        return "1. Payment is due within 30 days of invoice date.\n" .
               "2. Late payments may incur additional charges.\n" .
               "3. All sales are final unless otherwise specified.\n" .
               "4. Warranty terms apply as per manufacturer specifications.\n" .
               "5. For any queries, please contact us within 7 days.";
    }

    /**
     * Reset to default settings.
     */
    public function resetDefaults()
    {
        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        if ($settings) {
            $settings->update([
                'invoice_message_template' => $this->getDefaultMessageTemplate(),
                'invoice_terms_conditions' => $this->getDefaultTermsConditions(),
                'invoice_header_text' => 'INVOICE',
                'invoice_footer_text' => 'Thank you for your business!',
                'invoice_template' => 'default'
            ]);
        }

        return redirect()->route('invoice-settings.index')
            ->with('success', 'Settings reset to defaults successfully!');
    }

    /**
     * Display WhatsApp management page.
     */
    public function whatsappManagement()
    {
        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        // Default settings if none exist
        if (!$settings) {
            $settings = new Setting([
                'tenant_id' => $tenant->id,
                'company_name' => 'Your Company Name',
                'currency' => 'BDT',
                'whatsapp_enabled' => true,
                'invoice_template' => 'default',
                'invoice_message_template' => $this->getDefaultMessageTemplate(),
                'invoice_terms_conditions' => $this->getDefaultTermsConditions(),
                'invoice_header_text' => 'INVOICE',
                'invoice_footer_text' => 'Thank you for your business!',
                'auto_send_whatsapp' => false,
                'include_company_logo' => true,
                'include_qr_code' => false,
                'whatsapp_country_code' => '+880',
                'whatsapp_welcome_message' => $this->getDefaultWelcomeMessage(),
                'whatsapp_payment_reminder_message' => $this->getDefaultPaymentReminderMessage(),
                'whatsapp_delivery_message' => $this->getDefaultDeliveryMessage(),
                'whatsapp_thank_you_message' => $this->getDefaultThankYouMessage(),
                'whatsapp_auto_format_numbers' => true,
                'whatsapp_retry_attempts' => 3,
                'whatsapp_include_business_hours' => false,
                'whatsapp_business_hours' => '9:00 AM - 6:00 PM'
            ]);
        }
        
        return view('whatsapp.management', compact('settings'));
    }

    /**
     * Generate WhatsApp QR code data
     */
    public function generateWhatsAppQr(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'message_type' => 'required|in:invoice,welcome,reminder,delivery,thank_you,custom',
            'data' => 'array'
        ]);

        $whatsappService = new \App\Services\WhatsAppService();
        $phone = $request->input('phone');
        $messageType = $request->input('message_type');
        $data = $request->input('data', []);

        try {
            switch ($messageType) {
                case 'invoice':
                    $result = $whatsappService->createInvoiceWhatsApp($phone, $data);
                    break;
                case 'welcome':
                    $result = $whatsappService->createWelcomeWhatsApp($phone, $data);
                    break;
                case 'reminder':
                    $result = $whatsappService->createPaymentReminderWhatsApp($phone, $data);
                    break;
                case 'delivery':
                    $result = $whatsappService->createDeliveryWhatsApp($phone, $data);
                    break;
                case 'thank_you':
                    $result = $whatsappService->createThankYouWhatsApp($phone, $data);
                    break;
                case 'custom':
                    $message = $data['message'] ?? 'Hello from ' . ($this->settings->company_name ?? 'our store');
                    $result = [
                        'phone' => $whatsappService->formatPhoneNumber($phone),
                        'message' => $message,
                        'whatsapp_url' => $whatsappService->generateWhatsAppUrl($phone, $message),
                        'qr_code_url' => $whatsappService->generateQrCodeUrl($phone, $message),
                        'qr_code_url_alt' => $whatsappService->generateQrCodeUrlAlt($phone, $message)
                    ];
                    break;
                default:
                    return response()->json(['success' => false, 'message' => 'Invalid message type']);
            }

            return response()->json([
                'success' => true,
                'data' => $result
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate WhatsApp QR: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display WhatsApp Web connection page
     */
    public function webConnect()
    {
        return view('whatsapp.web-connect');
    }

    public function whatsappConnect()
    {
        return view('whatsapp.connect');
    }

    /**
     * Check WhatsApp connection status
     */
    public function checkWhatsAppConnection(Request $request)
    {
        try {
            $tenant = app('tenant');
            $settings = Setting::where('tenant_id', $tenant->id)->first();
            
            if (!$settings) {
                return response()->json([
                    'connected' => false,
                    'message' => 'Settings not found'
                ]);
            }
            
            // Check if WhatsApp is connected
            $isConnected = $settings->whatsapp_connected == 1;
            
            if ($isConnected) {
                // Return connection info
                return response()->json([
                    'connected' => true,
                    'device_info' => [
                        'phone' => $settings->whatsapp_connected_phone ?? '+880XXXXXXXXX',
                        'device_name' => $settings->whatsapp_device_name ?? 'Mobile Device',
                        'connected_at' => $settings->whatsapp_connected_at ?? now()->format('Y-m-d H:i:s'),
                        'session_id' => $settings->whatsapp_session_id ?? 'session_' . time()
                    ],
                    'last_checked' => now()->toISOString()
                ]);
            } else {
                // Simulate connection attempt - in real implementation this would check actual WhatsApp Web status
                // For demo purposes, randomly connect after some time with QR scan simulation
                if ($request->has('qr_scanned') || ($request->has('simulate_connect') && rand(1, 100) <= 10)) {
                    // Simulate successful connection after QR scan
                    $deviceInfo = [
                        'phone' => '+880' . rand(1000000000, 9999999999),
                        'device_name' => 'Phone (' . strtoupper(substr(md5(time()), 0, 4)) . ')',
                        'connected_at' => now()->format('Y-m-d H:i:s'),
                        'session_id' => 'wa_session_' . time() . '_' . rand(1000, 9999)
                    ];
                    
                    // Update settings
                    $settings->update([
                        'whatsapp_connected' => 1,
                        'whatsapp_connected_phone' => $deviceInfo['phone'],
                        'whatsapp_device_name' => $deviceInfo['device_name'],
                        'whatsapp_connected_at' => $deviceInfo['connected_at'],
                        'whatsapp_session_id' => $deviceInfo['session_id']
                    ]);
                    
                    \Log::info('WhatsApp Web connection established', $deviceInfo);
                    
                    return response()->json([
                        'connected' => true,
                        'device_info' => $deviceInfo,
                        'message' => 'WhatsApp Web connected successfully!',
                        'just_connected' => true
                    ]);
                }
                
                return response()->json([
                    'connected' => false,
                    'message' => 'WhatsApp not connected. Please scan QR code.',
                    'last_checked' => now()->toISOString()
                ]);
            }
            
        } catch (\Exception $e) {
            \Log::error('WhatsApp connection check failed: ' . $e->getMessage());
            
            return response()->json([
                'connected' => false,
                'error' => true,
                'message' => 'Failed to check connection status'
            ], 500);
        }
    }

    /**
     * Send test WhatsApp message
     */
    public function sendTestMessage(Request $request)
    {
        $request->validate([
            'message' => 'required|string|max:1000'
        ]);

        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        if (!$settings || !($settings->whatsapp_connected ?? false)) {
            return response()->json([
                'success' => false,
                'message' => 'WhatsApp not connected'
            ]);
        }

        try {
            // In real implementation, you would send the message through WhatsApp Web API
            // For now, we'll simulate success
            
            Log::info('WhatsApp test message sent', [
                'tenant_id' => $tenant->id,
                'message' => $request->message,
                'phone' => $settings->whatsapp_connected_phone ?? 'Unknown'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Test message sent successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send message: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Disconnect WhatsApp
     */
    public function disconnectWhatsApp()
    {
        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        if ($settings) {
            $settings->update([
                'whatsapp_connected' => false,
                'whatsapp_connected_phone' => null,
                'whatsapp_device_name' => null,
                'whatsapp_connected_at' => null,
                'whatsapp_session_id' => null
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'WhatsApp disconnected successfully'
        ]);
    }

    /**
     * Send WhatsApp message for invoice
     */
    public function sendInvoiceMessage(Request $request)
    {
        // Different validation rules based on request type
        if ($request->has('id')) {
            // Request from send button - only need phone and id
            $request->validate([
                'phone' => 'required|string',
                'id' => 'required|integer',
                'type' => 'required|string'
            ]);
        } else {
            // Manual request - need all fields
            $request->validate([
                'phone' => 'required|string',
                'customer_name' => 'required|string',
                'invoice_number' => 'required|string',
                'total_amount' => 'required|numeric',
                'items' => 'required|string'
            ]);
        }

        $tenant = app('tenant');
        $settings = Setting::where('tenant_id', $tenant->id)->first();
        
        // TEMPORARILY DISABLED FOR TESTING - UNCOMMENT WHEN WHATSAPP IS CONNECTED
        // if (!$settings || !($settings->whatsapp_connected ?? false)) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'WhatsApp not connected. Please connect your WhatsApp first.',
        //         'redirect_url' => route('whatsapp.connect')
        //     ]);
        // }
        
        // Create default settings if none exist
        if (!$settings) {
            $settings = new Setting([
                'tenant_id' => $tenant->id,
                'currency' => 'BHD',
                'company_name' => 'Your Company Name'
            ]);
        }

        try {
            $whatsappService = new \App\Services\WhatsAppService();
            
            // Get sales data if ID is provided
            $sale = null;
            if ($request->has('id') && $request->id) {
                $sale = \App\Models\Sale::with(['customer', 'sellTransactions', 'sellTransactions.product.ProductBrand', 'sellTransactions.product.ProductModel', 'exchanges.brand', 'exchanges.mobileModel'])
                    ->where('tenant_id', $tenant->id)->find($request->id);
                
                if (!$sale) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Sale not found with ID: ' . $request->id
                    ], 404);
                }
            }
            
            // Build invoice data from sale or request
            if ($sale) {
                // Calculate customer's total due amount if available
                $customerTotalDue = 0;
                if ($sale->customer) {
                    $totalDueFromSales = \App\Models\Sale::where('customer_id', $sale->customer->id)
                        ->where('return_status', '!=', 'returned')
                        ->sum('due_amount');
                    $advances = \App\Models\CustomerPayment::where('customer_id', $sale->customer->id)
                        ->where('type', 'advance')
                        ->sum('amount');
                    $customerTotalDue = max(0, $totalDueFromSales - $advances);
                }
                
                // Fix customer name to avoid duplication with phone
                $customerName = $sale->customer->name ?? 'Customer';
                $customerPhone = $sale->customer->phone ?? $request->phone;
                
                $invoiceData = [
                    'customer_name' => $customerName,
                    'customer_phone' => $customerPhone,
                    'invoice_number' => $sale->id,
                    'invoice_date' => $sale->created_at->format('m/d/Y'),
                    'item_list' => $this->formatItemList($sale),
                    'total_amount' => number_format($sale->total_amount, 2),
                    'paid_amount' => number_format($sale->paid_amount, 2),
                    'due_amount' => number_format($sale->due_amount, 2),
                    'customer_total_due' => number_format($customerTotalDue, 2),
                    'currency' => 'BHD', // Force BHD currency for Bahrain
                    'payment_method' => $sale->payment_method ?? 'cash',
                    'payment_status' => $sale->payment_status ?? 'paid',
                    'invoice_link' => url("/sales/{$sale->id}/table-invoice")
                ];
                $phone = $sale->customer->phone ?? $request->phone;
            } else {
                $invoiceData = [
                    'customer_name' => $request->customer_name ?? 'Customer',
                    'customer_phone' => $request->phone,
                    'invoice_number' => $request->invoice_number ?? 'N/A',
                    'invoice_date' => date('m/d/Y'),
                    'item_list' => $request->items ?? 'No items',
                    'total_amount' => number_format($request->total_amount ?? 0, 2),
                    'currency' => $settings->currency ?? 'BDT',
                    'payment_method' => $request->payment_method ?? 'cash',
                    'payment_status' => $request->payment_status ?? 'paid',
                    'invoice_link' => $request->invoice_link ?? '#'
                ];
                $phone = $request->phone;
            }

            // Create WhatsApp data
            $whatsappData = $whatsappService->createInvoiceWhatsApp($phone, $invoiceData);
            
            // If only URL requested, return it
            if ($request->get_url_only) {
                return response()->json([
                    'success' => true,
                    'message' => 'WhatsApp URL generated successfully!',
                    'whatsapp_url' => $whatsappData['whatsapp_url'],
                    'qr_code_url' => $whatsappData['qr_code_url'],
                    'phone' => $whatsappData['phone'],
                    'preview_message' => $whatsappData['message']
                ]);
            }
            
            // Log the action
            Log::info('WhatsApp invoice URL generated', [
                'tenant_id' => $tenant->id,
                'phone' => $whatsappData['phone'],
                'invoice_number' => $invoiceData['invoice_number'],
                'customer_name' => $invoiceData['customer_name']
            ]);

            return response()->json([
                'success' => true,
                'message' => 'WhatsApp URL generated successfully!',
                'whatsapp_url' => $whatsappData['whatsapp_url'],
                'qr_code_url' => $whatsappData['qr_code_url'],
                'phone' => $whatsappData['phone'],
                'preview_message' => $whatsappData['message']
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send invoice message: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get default welcome message.
     */
    private function getDefaultWelcomeMessage()
    {
        return "🎉 *Welcome to {company_name}!*\n\n" .
               "Hello {customer_name}! 👋\n\n" .
               "Thank you for choosing us for your mobile phone needs. We're excited to serve you!\n\n" .
               "📱 For any assistance, feel free to contact us.\n" .
               "📞 Phone: {customer_phone}\n\n" .
               "Best regards,\n{company_name} Team 🙏";
    }

    /**
     * Get default payment reminder message.
     */
    private function getDefaultPaymentReminderMessage()
    {
        return "💳 *Payment Reminder*\n\n" .
               "Dear {customer_name},\n\n" .
               "This is a friendly reminder that your invoice #{invoice_number} for {currency} {total_amount} is due.\n\n" .
               "📅 *Due Date:* {due_date}\n" .
               "💰 *Amount:* {currency} {total_amount}\n\n" .
               "Please make the payment at your earliest convenience.\n\n" .
               "🔗 *View Invoice:* {invoice_link}\n\n" .
               "Thank you for your business! 🙏";
    }

    /**
     * Format item list for WhatsApp message
     */
    private function formatItemList($sale)
    {
        $items = [];
        $currency = 'BHD'; // Force BHD currency for Bahrain
        
        // Check if sale has sellTransactions (line items) - using exact same logic as invoice print
        if ($sale->sellTransactions && $sale->sellTransactions->count() > 0) {
            foreach ($sale->sellTransactions as $transaction) {
                // Use exact same fields as invoice print
                $productName = $transaction->name ?? 'Product';
                $quantity = $transaction->sold_quantity ?? 1;
                $unitPrice = $transaction->sell_price ?? 0;
                
                // Ensure we have valid numbers
                if (!is_numeric($unitPrice)) $unitPrice = 0;
                if (!is_numeric($quantity)) $quantity = 1;
                
                $price = number_format($unitPrice, 2);
                $lineTotal = number_format($unitPrice * $quantity, 2);
                
                $items[] = "• {$productName} x{$quantity} @ {$currency} {$price} = {$currency} {$lineTotal}";
            }
        }
        
        // Check for exchanges too (like the invoice does)
        if ($sale->exchanges && $sale->exchanges->count() > 0) {
            foreach ($sale->exchanges as $exchange) {
                $exchangeName = "Exchanged: " . ($exchange->brand->name ?? 'N/A') . " " . ($exchange->mobileModel->name ?? 'N/A');
                $exchangePrice = $exchange->price ?? 0;
                $price = number_format($exchangePrice, 2);
                
                $items[] = "• {$exchangeName} x1 @ {$currency} {$price} = {$currency} {$price}";
            }
        }
        
        // If no items found from transactions, create from sale total
        if (empty($items)) {
            $total = is_numeric($sale->total_amount) ? number_format($sale->total_amount, 2) : '0.00';
            // Try to use any available product info from the sale itself
            $productName = $sale->product_name ?? 'Sale Item';
            $items[] = "• {$productName} x1 @ {$currency} {$total} = {$currency} {$total}";
        }
        
        return implode("\n", $items);
    }

    /**
     * Get default delivery message.
     */
    private function getDefaultDeliveryMessage()
    {
        return "🚚 *Order Delivery Update*\n\n" .
               "Dear {customer_name},\n\n" .
               "Great news! Your order #{invoice_number} is ready for delivery! 📦\n\n" .
               "📱 *Items:*\n{item_list}\n\n" .
               "📍 *Delivery Address:* [To be confirmed]\n" .
               "🕐 *Estimated Delivery:* [To be confirmed]\n\n" .
               "Our delivery team will contact you shortly.\n\n" .
               "Thank you for choosing {company_name}! 🙏";
    }

    /**
     * Get default thank you message.
     */
    private function getDefaultThankYouMessage()
    {
        return "🎉 *Thank You for Your Purchase!*\n\n" .
               "Dear {customer_name},\n\n" .
               "Thank you for choosing {company_name}! Your purchase has been completed successfully. ✅\n\n" .
               "🧾 *Invoice:* #{invoice_number}\n" .
               "💰 *Amount:* {currency} {total_amount}\n" .
               "📅 *Date:* {invoice_date}\n\n" .
               "We hope you enjoy your new device! 📱\n\n" .
               "⭐ Please rate your experience with us.\n" .
               "📞 For any support, contact us anytime.\n\n" .
               "Best regards,\n{company_name} Team 🙏";
    }
}