<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use App\Models\Location;
use App\Models\MobileModel;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;


class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $tenant = app('tenant');

            if ($request->ajax()) {
                $query = Product::where('tenant_id', $tenant->id);

                // Apply search filter
                if ($request->has('search') && !empty($request->search)) {
                    $searchTerm = $request->search;
                    $query->where(function($q) use ($searchTerm) {
                        $q->where('brand', 'LIKE', "%{$searchTerm}%")
                          ->orWhere('model', 'LIKE', "%{$searchTerm}%")
                          ->orWhere('imei_or_sn', 'LIKE', "%{$searchTerm}%")
                          ->orWhere('condition', 'LIKE', "%{$searchTerm}%")
                          ->orWhere('color', 'LIKE', "%{$searchTerm}%");
                    });
                }

            // Apply date filter
            if ($request->has('date_from') && !empty($request->date_from)) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }
            if ($request->has('date_to') && !empty($request->date_to)) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            // Apply type filter
            if ($request->has('type_filter') && !empty($request->type_filter)) {
                $query->where('type', $request->type_filter);
            }

            // Apply condition filter
            if ($request->has('condition_filter') && !empty($request->condition_filter)) {
                $query->where('condition', $request->condition_filter);
            }

            // Apply stock filter
            if ($request->has('stock_filter') && !empty($request->stock_filter)) {
                switch ($request->stock_filter) {
                    case 'in_stock':
                        $query->where('stock', '>', 0);
                        break;
                    case 'out_of_stock':
                        $query->where('stock', '<=', 0);
                        break;
                    case 'low_stock':
                        $query->where('stock', '>', 0)->where('stock', '<=', 5);
                        break;
                }
            }

            $totalRecords = Product::where('tenant_id', $tenant->id)->count();
            $totalPhones = Product::where('tenant_id', $tenant->id)->where('type', 'Phone')->count();
            $totalAccessories = Product::where('tenant_id', $tenant->id)->where('type', 'Accessories')->count();
            
            $products = $query->get();
            $filteredRecords = $products->count();

            $data = [];
            foreach ($products as $index => $product) {
                $stock = $product->stock ?? 0;
                $condition = $product->condition ?? 'unknown';

                // Get brand name - check if it's ID or name
                $brandName = $product->brand;
                if (is_numeric($brandName)) {
                    // If it's numeric, try to get brand name from brands table
                    $brand = Brand::find($brandName);
                    $brandName = $brand ? $brand->name : 'Unknown Brand';
                }

                // Get model name - check if it's ID or name  
                $modelName = $product->model;
                if (is_numeric($modelName)) {
                    // If it's numeric, try to get model name from mobile_models table
                    $mobileModel = MobileModel::find($modelName);
                    $modelName = $mobileModel ? $mobileModel->name : 'Unknown Model';
                }

                $stockBadge = '';
                if ($stock <= 0) {
                    $stockBadge = '<span class="badge badge-danger">Out of Stock</span>';
                } elseif ($stock <= 5) {
                    $stockBadge = '<span class="badge badge-warning">Low Stock</span>';
                } else {
                    $stockBadge = '<span class="badge badge-success">In Stock</span>';
                }

                $conditionBadge = '';
                switch($condition) {
                    case 'new':
                        $conditionBadge = '<span class="badge badge-success">New</span>';
                        break;
                    case 'used':
                        $conditionBadge = '<span class="badge badge-warning">Used</span>';
                        break;
                    case 'refurbished':
                        $conditionBadge = '<span class="badge badge-info">Refurbished</span>';
                        break;
                    default:
                        $conditionBadge = '<span class="badge badge-secondary">' . ucfirst($condition) . '</span>';
                }

                // Determine product type based on model name
                $productType = $product->type ?? 'Phone'; // Use actual type from database
                if (!$productType) {
                    // Fallback logic if type is empty
                    $model = $product->model ?? '';
                    if (stripos($model, 'case') !== false || 
                        stripos($model, 'charger') !== false || 
                        stripos($model, 'airpod') !== false ||
                        stripos($model, 'headphone') !== false) {
                        $productType = 'Accessories';
                    } else {
                        $productType = 'Phone';
                    }
                }

                $data[] = [
                    'checkbox' => '<input type="checkbox" class="product-checkbox" value="' . $product->id . '" data-product-name="' . htmlspecialchars($product->name ?? 'N/A') . '">',
                    'id' => $product->id,
                    'sl' => $index + 1,
                    'type' => '<span class="badge badge-primary">' . $productType . '</span>',
                    'name' => '<strong>' . ($product->name ?? 'N/A') . '</strong>',
                    'brand' => '<span class="badge badge-info">' . $brandName . '</span>',
                    'model' => '<span class="text-dark">' . $modelName . '</span>',
                    'imei_or_sn' => $product->imei_or_sn ?: '<em class="text-muted">N/A</em>',
                    'condition' => $conditionBadge,
                    'stock' => $stock . ' ' . $stockBadge,
                    'buy_price' => '<span class="text-success">BHD ' . number_format($product->buy_price ?? 0, 2) . '</span>',
                    'sell_price' => '<span class="text-info">BHD ' . number_format($product->sell_price ?? 0, 2) . '</span>',
                    'created_at' => $product->created_at ? $product->created_at->format('d M, Y') : 'N/A',
                    'actions' => '
                        <div class="btn-group">
                            <a href="' . route('product.editSimple', $product->id) . '" class="btn btn-sm btn-success" title="Quick Edit (Name, Buy Price, Stock)">
                                <i class="fas fa-bolt"></i>
                            </a>
                            <button class="btn btn-sm btn-primary" onclick="editProduct(' . $product->id . ')" title="Full Edit">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-danger" onclick="deleteProduct(' . $product->id . ')" title="Delete Product">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    ',
                ];
            }

            return response()->json([
                'draw' => intval($request->get('draw', 1)),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $data,
                'totalPhones' => $totalPhones,
                'totalAccessories' => $totalAccessories
            ]);
        }

        // Get filter options for dropdowns
        $models = MobileModel::where('tenant_id', $tenant->id)->orderBy('name')->get();
        $types = Product::where('tenant_id', $tenant->id)->whereNotNull('type')
            ->where('type', '!=', '')
            ->distinct()
            ->pluck('type');
        $conditions = Product::where('tenant_id', $tenant->id)->whereNotNull('condition')
            ->where('condition', '!=', '')
            ->distinct()
            ->pluck('condition');

        // Get counts for initial display
        $totalProducts = Product::where('tenant_id', $tenant->id)->count();
        $totalPhones = Product::where('tenant_id', $tenant->id)->where('type', 'Phone')->count();
        $totalAccessories = Product::where('tenant_id', $tenant->id)->where('type', 'Accessories')->count();
        
        // Get stock counts
        $stockPhones = Product::where('tenant_id', $tenant->id)->where('type', 'Phone')->sum('stock');
        $stockAccessories = Product::where('tenant_id', $tenant->id)->where('type', 'Accessories')->sum('stock');

        return view('product.index', compact('models', 'types', 'conditions', 'totalProducts', 'totalPhones', 'totalAccessories', 'stockPhones', 'stockAccessories'));
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json(['error' => $e->getMessage()], 500);
            }
            return back()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $tenant = app('tenant');
        $brands = Brand::where('tenant_id', $tenant->id)->get();
        return view('product.create', compact(['brands']));
    }
    public function storeDirectProduct(Request $request)
    {
        // Validate the incoming data
        $request->validate([
            'ProductType' => 'required|string',  // 'ProductType' comes from the form (product type)
            'product_name' => 'required|string', // 'product_name' comes from the form (model name)
            'item_price' => 'required|numeric',  // 'item_price' comes from the form (buy price)
            'stock_quantity' => 'required|integer', // 'stock_quantity' comes from the form (stock)
        ]);

        $tenant = app('tenant');

        // Create unique product name to allow multiple products with same base name
        $baseName = $request->product_name;
        $uniqueName = $baseName;
        $counter = 1;
        
        // Check if product name already exists and make it unique
        while (Product::where('tenant_id', $tenant->id)->where('name', $uniqueName)->exists()) {
            $counter++;
            $uniqueName = $baseName . ' (' . $counter . ')';
        }

        // Create and insert the data directly into the products table
        $product = Product::create([
            'tenant_id' => $tenant->id,
            'type' => $request->ProductType,  // Store the product type (brand)
            'name' => $uniqueName,  // Store the unique model name
            'buy_price' => $request->item_price,  // Store the buy price
            'stock' => $request->stock_quantity,  // Store the stock quantity
        ]);

        // Redirect back with success message
        return redirect()->back()->with('success', 'Product added successfully!');
    }

    /**
     * Store a newly created resource in storage.
     */

    public function store(Request $request)
    {
        try {
            // Validate the request data
            $validatedData = $request->validate([
                'customer_info' => 'nullable|integer',
                'brand' => 'required|integer',
                'model' => 'required|integer',
                'imei_or_sn' => 'required|string|max:255',
                'condition' => 'required|string|max:50',
                'color' => 'required|string|max:50',
                'gb' => 'required|integer',
                'buy_price' => 'required|numeric',
                'stock' => 'required|integer',
            ]);

            // Generate product name from brand and model
            $brand = Brand::find($validatedData['brand']);
            $model = MobileModel::find($validatedData['model']);
            $productName = ($brand ? $brand->name : 'Unknown Brand') . ' ' . ($model ? $model->name : 'Unknown Model');

            $tenant = app('tenant');

            // Store the data in the database (no sell_price needed)
            $product = Product::create([
                'tenant_id' => $tenant->id,
                'customer_info' => $validatedData['customer_info'],
                'brand' => $validatedData['brand'],
                'model' => $validatedData['model'],
                'name' => $productName,  // Use generated name from brand + model
                'type' => 'Phone',  // Default type for mobile products
                'imei_or_sn' => $validatedData['imei_or_sn'],
                'condition' => $validatedData['condition'],
                'color' => $validatedData['color'],
                'gb' => $validatedData['gb'],
                'buy_price' => $validatedData['buy_price'],
                'stock' => $validatedData['stock'],
            ]);

            // Return a success response
            // return response()->json([
            //     'message' => 'Product added successfully',
            //     'product' => $product
            // ], 201); // HTTP 201 Created
            return redirect()->back()->with('success', $product->id);
        } catch (\Exception $e) {
            // Handle errors and exceptions
            return response()->json([
                'message' => 'An error occurred while adding the product',
                'error' => $e->getMessage()
            ], 500); // HTTP 500 Internal Server Error
        }
    }


    public function product_stock(Request $request)
    {
        $tenant = app('tenant');
        $products = Product::with(['Customer'])->where('tenant_id', $tenant->id)->get();
        // Calculate total cost (buy_price * stock)
        $totalCost = $products->sum(function ($product) {
            return $product->buy_price * $product->stock;
        });

        $totalStock = $products->sum(function ($product) {
            return $product->stock;
        });

        $totalNewStock = $products->filter(function ($product) {
            return $product->condition === 'new'; // Filter only 'new' products
        })->sum('stock');

        $totalOldStock = $products->filter(function ($product) {
            return $product->condition === 'old'; // Filter only 'new' products
        })->sum('stock');

        // dd($totalNewStock);

        return view('product.stock', compact(['products', 'totalCost', 'totalStock', 'totalNewStock', 'totalOldStock']));
    }

    public function fetchProducts(Request $request)
    {
        $searchType = $request->input('search_type', 'all');
        $searchValue = $request->input('search_value', '');

        $tenant = app('tenant');
        $query = Product::with(['Customer'])->where('tenant_id', $tenant->id)->where('stock', '>', 0);

        if ($searchType !== 'all' && !empty($searchValue)) {
            if ($searchType == 'model') {
                $query->where('model', 'LIKE', "%$searchValue%");
            } elseif ($searchType == 'brand') {
                $query->where('brand', 'LIKE', "%$searchValue%");
            } elseif ($searchType == 'imei') {
                $query->where('imei_or_sn', 'LIKE', "%$searchValue%");
            } elseif ($searchType == 'name') {
                $query->where('name', 'LIKE', "%$searchValue%");
            }
        }

        $products = $query->get();

        return response()->json($products);
    }



    public function print_receipt($id)
    {
        // return $id;
        $tenant = app('tenant');
        $product = Product::with(['Customer', 'ProductBrand', 'ProductModel'])->where('tenant_id', $tenant->id)->findOrFail($id);
        $location = Location::first(); // Get the first location since locations don't have tenant_id
        return view('invoice.receipt', compact(['product', 'location']));
    }


    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        $tenant = app('tenant');
        $product = Product::with(['Customer', 'ProductBrand', 'ProductModel'])->where('tenant_id', $tenant->id)->findOrFail($product->id);
        return view('product.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $tenant = app('tenant');
        $product = Product::where('tenant_id', $tenant->id)->findOrFail($product->id);
        $brands = Brand::where('tenant_id', $tenant->id)->get();
        return view('product.edit', compact(['product', 'brands']));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        try {
            $tenant = app('tenant');
            $product = Product::where('tenant_id', $tenant->id)->findOrFail($product->id);

            // Validate the request data
            $validatedData = $request->validate([
                'customer_info' => 'nullable|integer',
                'brand' => 'required|integer',
                'model' => 'required|integer',
                'imei_or_sn' => 'required|string|max:255',
                'condition' => 'required|string|max:50',
                'color' => 'required|string|max:50',
                'gb' => 'required|integer',
                'buy_price' => 'required|numeric',
                'stock' => 'required|integer',
            ]);

            // Generate product name from brand and model
            $brand = Brand::find($validatedData['brand']);
            $model = MobileModel::find($validatedData['model']);
            $productName = ($brand ? $brand->name : 'Unknown Brand') . ' ' . ($model ? $model->name : 'Unknown Model');

            // Update the product in the database
            $product->update([
                'customer_info' => $validatedData['customer_info'],
                'brand' => $validatedData['brand'],
                'model' => $validatedData['model'],
                'name' => $productName,  // Use generated name from brand + model
                'type' => 'Phone',  // Default type for mobile products
                'imei_or_sn' => $validatedData['imei_or_sn'],
                'condition' => $validatedData['condition'],
                'color' => $validatedData['color'],
                'gb' => $validatedData['gb'],
                'buy_price' => $validatedData['buy_price'],
                'stock' => $validatedData['stock'],
            ]);

            // Redirect with success message
            return redirect()->route('product.index')->with('success', "Product ID {$product->id} successfully updated!");
        } catch (\Exception $e) {
            // Redirect with error message
            return redirect()->back()->with('error', 'An error occurred while updating the product: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        $tenant = app('tenant');
        $product = Product::where('tenant_id', $tenant->id)->findOrFail($product->id);
        $product->delete();
        return redirect()->back()->with('success', "Product successfully deleted!");
    }

    /**
     * Bulk delete multiple products
     */
    public function bulkDelete(Request $request)
    {
        try {
            $tenant = app('tenant');
            
            // Validate the request
            $request->validate([
                'product_ids' => 'required|array|min:1',
                'product_ids.*' => 'integer|exists:products,id'
            ]);

            $productIds = $request->input('product_ids');
            
            // Ensure products belong to current tenant
            $products = Product::where('tenant_id', $tenant->id)
                ->whereIn('id', $productIds)
                ->get();

            if ($products->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No valid products found for deletion.'
                ], 404);
            }

            $deletedCount = 0;
            $productNames = [];

            foreach ($products as $product) {
                $productNames[] = $product->name ?? "Product #{$product->id}";
                $product->delete();
                $deletedCount++;
            }

            return response()->json([
                'success' => true,
                'message' => "Successfully deleted {$deletedCount} product(s).",
                'deleted_count' => $deletedCount,
                'deleted_products' => $productNames
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting products: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search for existing products by name for auto-complete
     */
    public function searchProducts(Request $request)
    {
        try {
            \Log::info('Product search endpoint hit', ['query' => $request->get('q', ''), 'user_id' => auth()->id()]);
            
            $tenant = app('tenant');
            $query = $request->get('q', '');

            \Log::info('Product search request', ['query' => $query, 'tenant_id' => $tenant->id]);

            if (strlen($query) < 2) {
                \Log::info('Query too short, returning empty array');
                return response()->json([]);
            }

            $products = Product::where('tenant_id', $tenant->id)
                ->where('name', 'LIKE', '%' . $query . '%')
                ->select('id', 'name', 'buy_price', 'stock', 'type')
                ->limit(10)
                ->get();

            \Log::info('Product search results', ['count' => $products->count(), 'products' => $products->toArray()]);

            return response()->json($products);
        } catch (\Exception $e) {
            \Log::error('Product search error: ' . $e->getMessage());
            \Log::error('Product search stack trace: ' . $e->getTraceAsString());
            return response()->json(['error' => 'Search failed: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Update stock for an existing product
     */
    public function updateStock(Request $request)
    {
        $tenant = app('tenant');

        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'additional_stock' => 'required|integer|min:1',
        ]);

        try {
            $product = Product::where('tenant_id', $tenant->id)
                ->findOrFail($request->product_id);

            $oldStock = $product->stock;
            $product->stock += $request->additional_stock;
            $product->save();

            return response()->json([
                'success' => true,
                'message' => 'Stock updated successfully!',
                'old_stock' => $oldStock,
                'additional_stock' => $request->additional_stock,
                'new_stock' => $product->stock,
                'product_name' => $product->name
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating stock: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display comprehensive product stock report
     */
    public function stockReport(Request $request)
    {
        try {
            $tenant = app('tenant');
            
            // Get all products with relationships
            $products = Product::with(['Customer'])->where('tenant_id', $tenant->id)->get();
            
            // Calculate comprehensive statistics
            $totalProducts = $products->count();
            $totalStock = $products->sum('stock');
            $totalBuyValue = $products->sum(function ($product) {
                return $product->buy_price * $product->stock;
            });
            $totalSellValue = $products->sum(function ($product) {
                return ($product->sell_price ?? $product->buy_price * 1.2) * $product->stock;
            });
            $potentialProfit = $totalSellValue - $totalBuyValue;
            
            // Stock by condition
            $newConditionStock = $products->where('condition', 'new')->sum('stock');
            $usedConditionStock = $products->where('condition', 'used')->sum('stock');
            $refurbishedConditionStock = $products->where('condition', 'refurbished')->sum('stock');
            $oldConditionStock = $products->where('condition', 'old')->sum('stock');
            
            // Stock by type
            $phoneStock = $products->where('type', 'Phone')->sum('stock');
            $accessoriesStock = $products->where('type', 'Accessories')->sum('stock');
            
            // Low stock products (stock <= 5)
            $lowStockProducts = $products->where('stock', '<=', 5)->where('stock', '>', 0);
            $outOfStockProducts = $products->where('stock', '<=', 0);
            
            // Brand-wise statistics
            $brandStats = $products->groupBy('brand')->map(function ($brandProducts) {
                return [
                    'count' => $brandProducts->count(),
                    'stock' => $brandProducts->sum('stock'),
                    'value' => $brandProducts->sum(function ($product) {
                        return $product->buy_price * $product->stock;
                    })
                ];
            });
            
            // Top 10 most valuable products by total value (buy_price * stock)
            $topValueProducts = $products->sortByDesc(function ($product) {
                return $product->buy_price * $product->stock;
            })->take(10);
            
            // Monthly stock movements (if you have created_at dates)
            $monthlyAdditions = $products->groupBy(function ($product) {
                return $product->created_at ? $product->created_at->format('Y-m') : 'Unknown';
            })->map(function ($monthProducts) {
                return [
                    'count' => $monthProducts->count(),
                    'stock' => $monthProducts->sum('stock'),
                    'value' => $monthProducts->sum(function ($product) {
                        return $product->buy_price * $product->stock;
                    })
                ];
            });
            
            // Stock status categories
            $inStockProducts = $products->where('stock', '>', 5)->count();
            $lowStockCount = $lowStockProducts->count();
            $outOfStockCount = $outOfStockProducts->count();
            
            // Average prices
            $avgBuyPrice = $products->where('buy_price', '>', 0)->avg('buy_price');
            $avgSellPrice = $products->where('sell_price', '>', 0)->avg('sell_price');
            
            // Stock value by condition
            $newConditionValue = $products->where('condition', 'new')->sum(function ($product) {
                return $product->buy_price * $product->stock;
            });
            $usedConditionValue = $products->where('condition', 'used')->sum(function ($product) {
                return $product->buy_price * $product->stock;
            });
            $refurbishedConditionValue = $products->where('condition', 'refurbished')->sum(function ($product) {
                return $product->buy_price * $product->stock;
            });
            $oldConditionValue = $products->where('condition', 'old')->sum(function ($product) {
                return $product->buy_price * $product->stock;
            });
            
            return view('product.stock-report', compact(
                'products',
                'totalProducts',
                'totalStock',
                'totalBuyValue',
                'totalSellValue',
                'potentialProfit',
                'newConditionStock',
                'usedConditionStock',
                'refurbishedConditionStock',
                'oldConditionStock',
                'phoneStock',
                'accessoriesStock',
                'lowStockProducts',
                'outOfStockProducts',
                'brandStats',
                'topValueProducts',
                'monthlyAdditions',
                'inStockProducts',
                'lowStockCount',
                'outOfStockCount',
                'avgBuyPrice',
                'avgSellPrice',
                'newConditionValue',
                'usedConditionValue',
                'refurbishedConditionValue',
                'oldConditionValue'
            ));
            
        } catch (\Exception $e) {
            return back()->with('error', 'Error generating stock report: ' . $e->getMessage());
        }
    }

    /**
     * Show simplified edit form for product name, price, and stock only
     */
    public function editSimple(Product $product)
    {
        $tenant = app('tenant');
        $product = Product::where('tenant_id', $tenant->id)->findOrFail($product->id);
        return view('product.edit-simple', compact('product'));
    }

    /**
     * Update product with simplified fields (name, sell_price, stock)
     */
    public function updateSimple(Request $request, Product $product)
    {
        try {
            $tenant = app('tenant');
            $product = Product::where('tenant_id', $tenant->id)->findOrFail($product->id);

            // Validate the request data for simple edit
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'buy_price' => 'required|numeric|min:0',
                'stock' => 'required|integer|min:0',
            ]);

            // Update only the specified fields
            $product->update([
                'name' => $validatedData['name'],
                'buy_price' => $validatedData['buy_price'],
                'stock' => $validatedData['stock'],
            ]);

            // Redirect with success message
            return redirect()->route('product.index')->with('success', "Product '{$product->name}' successfully updated!");
        } catch (\Exception $e) {
            // Redirect with error message
            return redirect()->back()->with('error', 'An error occurred while updating the product: ' . $e->getMessage());
        }
    }

}
