<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\SellTransaction;
use App\Models\Sale;
use App\Models\Setting;
use App\Models\Brand;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SellTransactionController extends Controller
{

    public function index(Request $request)
    {
        $tenant = app('tenant');

        if ($request->ajax()) {
            $query = SellTransaction::with(['product', 'sale', 'sale.customer'])
                ->where('tenant_id', $tenant->id);

            // Apply search filter
            if ($request->has('search') && !empty($request->search)) {
                $searchTerm = $request->search;
                $query->where(function($q) use ($searchTerm) {
                    $q->where('name', 'LIKE', "%{$searchTerm}%")
                      ->orWhere('type', 'LIKE', "%{$searchTerm}%")
                      ->orWhere('payment_method', 'LIKE', "%{$searchTerm}%")
                      ->orWhereHas('product', function($product) use ($searchTerm) {
                          $product->where('name', 'LIKE', "%{$searchTerm}%")
                                  ->orWhere('imei_or_sn', 'LIKE', "%{$searchTerm}%");
                      })
                      ->orWhereHas('sale.customer', function($customer) use ($searchTerm) {
                          $customer->where('name', 'LIKE', "%{$searchTerm}%");
                      });
                });
            }

            // Apply date filter
            if ($request->has('date_from') && !empty($request->date_from)) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }
            if ($request->has('date_to') && !empty($request->date_to)) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            // Apply type filter
            if ($request->has('type_filter') && !empty($request->type_filter)) {
                $query->where('type', $request->type_filter);
            }

            // Apply payment method filter
            if ($request->has('payment_filter') && !empty($request->payment_filter)) {
                $query->where('payment_method', $request->payment_filter);
            }

            $totalRecords = SellTransaction::where('tenant_id', $tenant->id)->count();
            $sales = $query->orderBy('created_at', 'desc')->get();
            $filteredRecords = $sales->count();

            // Get settings for currency
            $all_settings = \App\Models\Setting::where('tenant_id', $tenant->id)->first();
            $currency = $all_settings->currency ?? 'BDT';

            $data = [];
            foreach ($sales as $index => $sale) {
                $profitBadge = '';
                if ($sale->profit > 100) {
                    $profitBadge = '<span class="badge badge-success">High Profit</span>';
                } elseif ($sale->profit > 0) {
                    $profitBadge = '<span class="badge badge-info">Profitable</span>';
                } else {
                    $profitBadge = '<span class="badge badge-danger">Loss</span>';
                }

                $paymentMethodBadge = '';
                switch($sale->payment_method) {
                    case 'cash':
                        $paymentMethodBadge = '<span class="badge badge-success">Cash</span>';
                        break;
                    case 'credit':
                        $paymentMethodBadge = '<span class="badge badge-warning">Credit</span>';
                        break;
                    case 'card':
                        $paymentMethodBadge = '<span class="badge badge-info">Card</span>';
                        break;
                    default:
                        $paymentMethodBadge = '<span class="badge badge-secondary">' . ucfirst($sale->payment_method) . '</span>';
                }

                $data[] = [
                    'id' => $sale->id,
                    'sl' => $index + 1,
                    'product_name' => '<strong>' . ($sale->product ? $sale->product->name : $sale->name) . '</strong>',
                    'customer_name' => $sale->sale && $sale->sale->customer ? $sale->sale->customer->name : '<em class="text-muted">Walk-in Customer</em>',
                    'type' => '<span class="badge badge-primary">' . ucfirst($sale->type) . '</span>',
                    'payment_method' => $paymentMethodBadge,
                    'sell_price' => '<span class="text-success">' . $currency . ' ' . number_format($sale->sell_price, 2) . '</span>',
                    'sold_quantity' => '<span class="badge badge-info">' . $sale->sold_quantity . '</span>',
                    'total_price' => '<span class="text-primary">' . $currency . ' ' . number_format($sale->sell_price * $sale->sold_quantity, 2) . '</span>',
                    'profit' => '<span class="text-success">' . $currency . ' ' . number_format($sale->profit, 2) . '</span> ' . $profitBadge,
                    'warranty' => $sale->warranty ?: '<em class="text-muted">No Warranty</em>',
                    'sale_date' => $sale->created_at->format('d M, Y H:i'),
                    'actions' => view('sell.partials.actions', compact('sale'))->render(),
                ];
            }

            // Calculate totals
            $totals = [
                'total_profit' => $sales->sum('profit'),
                'total_quantity' => $sales->sum('sold_quantity'),
                'total_amount' => $sales->sum(fn($s) => $s->sell_price * $s->sold_quantity)
            ];

            return response()->json([
                'data' => $data,
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'totals' => $totals
            ]);
        }

        $query = SellTransaction::with('product');

        $sales = $query->orderBy('created_at', 'desc')->get();

        $totals = [
            'total_profit' => $sales->sum('profit'),
            'total_sale' => $sales->sum('sold_quantity'),
            'total_amount' => $sales->sum(fn($sale) => $sale->sell_price * $sale->sold_quantity)
        ];

        // Get filter options
        $types = SellTransaction::where('tenant_id', $tenant->id)->distinct()->pluck('type')->filter();
        $paymentMethods = SellTransaction::where('tenant_id', $tenant->id)->distinct()->pluck('payment_method')->filter();

        // Get settings for currency
        $all_settings = \App\Models\Setting::where('tenant_id', $tenant->id)->first();

        return view('sell.index', compact('sales', 'totals', 'types', 'paymentMethods', 'all_settings'));
    }

    public function showSaleForm($id)
    {
        $product = Product::with(['Customer'])->findOrFail($id);
        return view('sell.form', compact('product'));
    }

    public function profitReport($id)
    {
        $product = Product::with(['Customer'])->findOrFail($id);
        return view('sell.form', compact('product'));
    }

    public function processSale(Request $request, $id)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'sell_price' => 'required|numeric|min:0',
            'sold_quantity' => 'required|integer|min:1',
            'warranty' => 'required|string',
        ]);

        $product = Product::with(['Customer'])->findOrFail($id);
        $customer = Customer::findOrFail($request->customer_id);

        if ($request->sold_quantity > $product->stock) {
            return back()->with('error', 'Not enough stock available. Available stock: ' . $product->stock);
        }

        try {
            $profit = ($request->sell_price - $product->buy_price) * $request->sold_quantity;

            $warranty_duration = $request->warranty;
            if ($request->warranty == 'Custom') {
                $warranty_duration = $request->custom_warranty_value . ' ' . $request->custom_warranty_unit;
            }

            // Create Sale record first
            $totalAmount = $request->sell_price * $request->sold_quantity;
            $paymentMethod = $request->payment_method ?? 'cash';
            $paymentStatus = $paymentMethod === 'credit' ? 'unpaid' : 'paid';
            $paidAmount = $paymentStatus === 'paid' ? $totalAmount : 0;
            $dueAmount = $paymentStatus === 'unpaid' ? $totalAmount : 0;

            $sale = Sale::create([
                'customer_id' => $request->customer_id,
                'total_amount' => $totalAmount,
                'discount' => 0,
                'net_amount' => $totalAmount,
                'paid_amount' => $paidAmount,
                'due_amount' => $dueAmount,
                'payment_method' => $paymentMethod,
                'payment_status' => $paymentStatus,
                'tenant_id' => app('tenant')->id ?? null,
            ]);

            // Create SellTransaction with sale_id
            $sellTransaction = SellTransaction::create([
                'sale_id' => $sale->id,
                'product_id' => $product->id,
                'customer_id' => $request->customer_id,
                'payment_method' => $request->payment_method ?? null,
                'sell_price' => $request->sell_price,
                'sold_quantity' => $request->sold_quantity,
                'profit' => $profit,
                'warranty' => $warranty_duration,
                'date_time' => now(),
                'type' => $product->type ?? 'Unknown',
                'name' => $product->name,
            ]);

            $product->decrement('stock', $request->sold_quantity);

            return view('sell.invoice', compact(['sellTransaction', 'product', 'customer']))->with('success', 'Sale recorded successfully!');
        } catch (\Exception $e) {
            Log::error('Error while storing sale in processSale: ' . $e->getMessage());
            return back()->with('error', 'An error occurred while processing the sale.');
        }
    }

    public function saleInvoice($id)
    {
        $sale = SellTransaction::with('product')->findOrFail($id);
        $all_settings = \App\Models\Setting::where('tenant_id', app('tenant')->id)->first();
        return view('sell.sale_invoice', compact('sale', 'all_settings'));
    }

    public function salesProfitReport(Request $request)
    {
        $transactions = SellTransaction::with([
            'product',
            'sale.customer'
        ]);

        // If it's an AJAX request with a date range
        if ($request->ajax() && $request->has('date_range')) {
            $dates = explode(' - ', $request->date_range);
            $startDate = Carbon::parse($dates[0])->startOfDay();
            $endDate = Carbon::parse($dates[1])->endOfDay();

            // Filter transactions within the selected date range
            $transactions->whereBetween('created_at', [$startDate, $endDate]);
        }

        $transactions = $transactions->get(); // Execute query

        if ($request->ajax()) {
            return response()->json(['transactions' => $transactions]);
        }

        return view('sell.profit_report', compact('transactions'));
    }



    public function destroy($sellTransaction)
    {
        // Find the transaction
        $sellTransaction = SellTransaction::findOrFail($sellTransaction);

        // Find the product related to the transaction
        $soldProduct = Product::findOrFail($sellTransaction->product_id);

        // Find the sale related to this transaction
        $sale = Sale::find($sellTransaction->sale_id);

        // Calculate the total quantity to restore in the stock
        $total_qty = $soldProduct->stock + $sellTransaction->sold_quantity;

        // Try to delete the transaction
        if ($sellTransaction->delete()) {
            // Only update the product stock if the deletion was successful
            $soldProduct->stock = $total_qty;
            $soldProduct->save();

            // If this was the last transaction for the sale, delete the sale and handle due amounts
            if ($sale && $sale->sellTransactions()->count() == 0) {
                // If the sale had due amount, we need to handle customer payments
                if ($sale->due_amount > 0 && $sale->customer_id) {
                    // Find and reverse any customer payments related to this sale
                    $customerPayments = \App\Models\CustomerPayment::where('tenant_id', app('tenant')->id ?? null)
                        ->whereNotNull('sale_ids')
                        ->get();
                    
                    foreach ($customerPayments as $payment) {
                        $saleIds = json_decode($payment->sale_ids, true);
                        if (is_array($saleIds) && in_array($sale->id, $saleIds)) {
                            // Remove this sale from the payment record
                            $saleIds = array_filter($saleIds, function($id) use ($sale) {
                                return $id != $sale->id;
                            });
                            
                            if (empty($saleIds)) {
                                // If no other sales in this payment, delete the payment record
                                $payment->delete();
                            } else {
                                // Update the payment record with remaining sale IDs
                                $payment->sale_ids = json_encode(array_values($saleIds));
                                $payment->save();
                            }
                        }
                    }
                }
                
                // Delete the sale
                $sale->delete();
            } else if ($sale) {
                // Recalculate sale totals if there are remaining transactions
                $remainingTransactions = $sale->sellTransactions();
                $totalAmount = $remainingTransactions->sum('sell_price');
                
                // Update sale totals
                $sale->total_amount = $totalAmount;
                if ($sale->paid_amount > $totalAmount) {
                    $sale->paid_amount = $totalAmount;
                    $sale->due_amount = 0;
                    $sale->payment_status = 'paid';
                } else {
                    $sale->due_amount = $totalAmount - $sale->paid_amount;
                    $sale->payment_status = $sale->due_amount > 0 ? 'unpaid' : 'paid';
                }
                $sale->save();
            }

            // Redirect with success message
            return redirect()->route('sale.index')->with('success', 'Successfully deleted!');
        } else {
            // In case the deletion fails
            return redirect()->route('sale.index')->with('error', 'Failed to delete transaction!');
        }
    }
    public function showForm($productId)

    {

        $product = Product::find($productId);

        $brands = Brand::all(); // Fetch all brands

        return view('sell.form', compact('product', 'brands'));
    }
}
