<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use ZipArchive;

class SystemUpdateController extends Controller
{
    // Current system version
    const CURRENT_VERSION = '1.0.0';
    
    // Update server URL - victorhost.com/posupdate
    const UPDATE_SERVER_URL = 'https://victorhost.com/posupdate';
    
    /**
     * Display the system update page
     */
    public function index()
    {
        $currentVersion = self::CURRENT_VERSION;
        $latestVersion = $this->checkForUpdates();
        $updateAvailable = version_compare($latestVersion, $currentVersion, '>');
        
        return view('system.update', compact('currentVersion', 'latestVersion', 'updateAvailable'));
    }
    
    /**
     * Check for available updates from victorhost.com/posupdate
     */
    public function checkForUpdates()
    {
        try {
            // Check update server for latest version
            $versionUrl = self::UPDATE_SERVER_URL . '/version.json';
            
            // Use file_get_contents with timeout
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10, // 10 seconds timeout
                    'ignore_errors' => true
                ]
            ]);
            
            $response = @file_get_contents($versionUrl, false, $context);
            
            if ($response === false) {
                Log::warning('Could not connect to update server');
                return self::CURRENT_VERSION;
            }
            
            $data = json_decode($response, true);
            
            if (isset($data['version'])) {
                Log::info('Latest version from server: ' . $data['version']);
                return $data['version'];
            }
            
            return self::CURRENT_VERSION;
        } catch (\Exception $e) {
            Log::error('Update check failed: ' . $e->getMessage());
            return self::CURRENT_VERSION;
        }
    }
    
    /**
     * Perform system update
     */
    public function update(Request $request)
    {
        try {
            $steps = [];
            
            // Step 1: Put application in maintenance mode
            $steps[] = $this->maintenanceMode(true);
            
            // Step 2: Backup current system
            $steps[] = $this->backupSystem();
            
            // Step 3: Download update package
            $updateFile = $this->downloadUpdate();
            $steps[] = ['step' => 'Download Update', 'status' => 'success', 'message' => 'Update package downloaded'];
            
            // Step 4: Extract update files
            $steps[] = $this->extractUpdate($updateFile);
            
            // Step 5: Run database migrations
            $steps[] = $this->runMigrations();
            
            // Step 6: Clear cache
            $steps[] = $this->clearCache();
            
            // Step 7: Turn off maintenance mode
            $steps[] = $this->maintenanceMode(false);
            
            return response()->json([
                'success' => true,
                'message' => 'System updated successfully to version ' . $this->checkForUpdates(),
                'steps' => $steps
            ]);
            
        } catch (\Exception $e) {
            // Restore from backup if update fails
            $this->restoreBackup();
            $this->maintenanceMode(false);
            
            Log::error('System update failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Update failed: ' . $e->getMessage(),
                'steps' => $steps ?? []
            ], 500);
        }
    }
    
    /**
     * Enable/disable maintenance mode
     */
    private function maintenanceMode($enable = true)
    {
        try {
            if ($enable) {
                Artisan::call('down', ['--render' => 'errors::503']);
                return ['step' => 'Maintenance Mode', 'status' => 'success', 'message' => 'System put in maintenance mode'];
            } else {
                Artisan::call('up');
                return ['step' => 'Maintenance Mode Off', 'status' => 'success', 'message' => 'System is back online'];
            }
        } catch (\Exception $e) {
            return ['step' => 'Maintenance Mode', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Backup current system
     */
    private function backupSystem()
    {
        try {
            $backupPath = storage_path('backups');
            if (!File::exists($backupPath)) {
                File::makeDirectory($backupPath, 0755, true);
            }
            
            $backupFile = $backupPath . '/backup_' . date('Y-m-d_H-i-s') . '.zip';
            
            // Create backup zip (simplified version)
            // In production, you'd want to backup database and files
            $zip = new ZipArchive();
            if ($zip->open($backupFile, ZipArchive::CREATE) === true) {
                // Add important files to backup
                $this->addFilesToZip($zip, base_path('app'), 'app');
                $this->addFilesToZip($zip, base_path('config'), 'config');
                $this->addFilesToZip($zip, base_path('database'), 'database');
                $this->addFilesToZip($zip, base_path('resources'), 'resources');
                $this->addFilesToZip($zip, base_path('routes'), 'routes');
                
                $zip->close();
            }
            
            return ['step' => 'Backup', 'status' => 'success', 'message' => 'System backed up successfully'];
        } catch (\Exception $e) {
            return ['step' => 'Backup', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Add files to zip recursively
     */
    private function addFilesToZip($zip, $folder, $zipFolder = '')
    {
        if (!File::exists($folder)) {
            return;
        }
        
        $files = File::allFiles($folder);
        foreach ($files as $file) {
            $relativePath = $zipFolder . '/' . $file->getRelativePathname();
            $zip->addFile($file->getRealPath(), $relativePath);
        }
    }
    
    /**
     * Download update package from victorhost.com/posupdate
     */
    private function downloadUpdate()
    {
        try {
            $latestVersion = $this->checkForUpdates();
            $updateUrl = self::UPDATE_SERVER_URL . '/updates/update-' . $latestVersion . '.zip';
            
            // Create updates directory if it doesn't exist
            $updatesDir = storage_path('updates');
            if (!File::exists($updatesDir)) {
                File::makeDirectory($updatesDir, 0755, true);
            }
            
            $updateFile = $updatesDir . '/update.zip';
            
            // Download update file from server
            $context = stream_context_create([
                'http' => [
                    'timeout' => 300, // 5 minutes timeout for large files
                    'ignore_errors' => true
                ]
            ]);
            
            $updateData = @file_get_contents($updateUrl, false, $context);
            
            if ($updateData === false) {
                throw new \Exception('Failed to download update from server');
            }
            
            file_put_contents($updateFile, $updateData);
            
            Log::info('Update downloaded successfully from: ' . $updateUrl);
            
            return $updateFile;
        } catch (\Exception $e) {
            Log::error('Download update failed: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Extract update files
     */
    private function extractUpdate($updateFile)
    {
        try {
            if (!File::exists($updateFile)) {
                throw new \Exception('Update file not found');
            }
            
            $zip = new ZipArchive();
            if ($zip->open($updateFile) === true) {
                $zip->extractTo(base_path());
                $zip->close();
                
                // Delete update file after extraction
                File::delete($updateFile);
                
                return ['step' => 'Extract Files', 'status' => 'success', 'message' => 'Update files extracted'];
            } else {
                throw new \Exception('Failed to open update package');
            }
        } catch (\Exception $e) {
            return ['step' => 'Extract Files', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Run database migrations
     */
    private function runMigrations()
    {
        try {
            Artisan::call('migrate', ['--force' => true]);
            return ['step' => 'Database Migration', 'status' => 'success', 'message' => 'Database updated successfully'];
        } catch (\Exception $e) {
            return ['step' => 'Database Migration', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Clear system cache
     */
    private function clearCache()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            
            return ['step' => 'Clear Cache', 'status' => 'success', 'message' => 'System cache cleared'];
        } catch (\Exception $e) {
            return ['step' => 'Clear Cache', 'status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Restore from backup
     */
    private function restoreBackup()
    {
        try {
            $backupPath = storage_path('backups');
            $backups = File::files($backupPath);
            
            if (empty($backups)) {
                return false;
            }
            
            // Get the latest backup
            usort($backups, function($a, $b) {
                return $b->getMTime() - $a->getMTime();
            });
            
            $latestBackup = $backups[0];
            
            $zip = new ZipArchive();
            if ($zip->open($latestBackup->getRealPath()) === true) {
                $zip->extractTo(base_path());
                $zip->close();
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            Log::error('Backup restore failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get update history/changelog from victorhost.com/posupdate
     */
    public function changelog()
    {
        try {
            // Fetch changelog from update server
            $changelogUrl = self::UPDATE_SERVER_URL . '/changelog.json';
            
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'ignore_errors' => true
                ]
            ]);
            
            $response = @file_get_contents($changelogUrl, false, $context);
            
            if ($response !== false) {
                $changelog = json_decode($response, true);
                
                if (is_array($changelog)) {
                    return response()->json($changelog);
                }
            }
            
            // Fallback to local changelog if server is unavailable
            $changelog = [
                [
                    'version' => '1.0.0',
                    'date' => '2025-10-21',
                    'changes' => [
                        'Initial release',
                        'POS system with sales management',
                        'Enhanced notification system with sounds',
                        'Auto-update system',
                        'Customer and inventory management'
                    ]
                ]
            ];
            
            return response()->json($changelog);
        } catch (\Exception $e) {
            Log::error('Changelog fetch failed: ' . $e->getMessage());
            
            // Return default changelog
            return response()->json([
                [
                    'version' => '1.0.0',
                    'date' => '2025-10-21',
                    'changes' => ['VECTOR MANAGEMENT POS System']
                ]
            ]);
        }
    }
}
