<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Sale extends Model
{
    use HasFactory;

    // Define the table associated with the model
    protected $table = 'sales';

    // Fillable fields for mass assignment
    protected $fillable = [
        'customer_id',
        'customer_name',
        'total_amount',
        'discount',
        'net_amount',
        'paid_amount',
        'due_amount',
        'payment_method',
        'payment_status',
        'return_status',
        'returned_at',
        'return_reason',
        'tenant_id',
    ];

    // Relationship with the Customer model (a sale belongs to a customer)
    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    // Relationship with the Payment model (a sale has many payments)
    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    // Relationship with the SellTransaction model (a sale has many sell transactions)
    public function sellTransactions()
    {
        return $this->hasMany(SellTransaction::class, 'sale_id', 'id');
    }



    // Relationship with Exchange model
    public function exchanges()
    {
        return $this->hasMany(Exchange::class);
    }

    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Mark this sale as returned
     */
    public function markAsReturned($reason = null)
    {
        $this->return_status = 'returned';
        $this->returned_at = now();
        $this->return_reason = $reason;
        
        // When returning, if there was a due amount, we need to clean it up
        if ($this->due_amount > 0 && $this->customer_id) {
            // Find and reverse any customer payments related to this sale
            $customerPayments = \App\Models\CustomerPayment::where('tenant_id', $this->tenant_id)
                ->whereNotNull('sale_ids')
                ->get();
            
            foreach ($customerPayments as $payment) {
                $saleIds = json_decode($payment->sale_ids, true);
                if (is_array($saleIds) && in_array($this->id, $saleIds)) {
                    // Remove this sale from the payment record
                    $saleIds = array_filter($saleIds, function($id) {
                        return $id != $this->id;
                    });
                    
                    if (empty($saleIds)) {
                        // If no other sales in this payment, delete the payment record
                        $payment->delete();
                    } else {
                        // Update the payment record with remaining sale IDs
                        $payment->sale_ids = json_encode(array_values($saleIds));
                        $payment->save();
                    }
                }
            }
        }
        
        // Set due amount to 0 for returned sales
        $this->due_amount = 0;
        $this->payment_status = 'returned';
        
        $this->save();
        
        return $this;
    }

    protected static function booted()
    {
        static::addGlobalScope('tenant', function ($builder) {
            if (app()->has('tenant')) {
                $builder->where('tenant_id', app('tenant')->id);
            }
        });

        // Handle sale deletion to clean up customer due amounts
        static::deleting(function ($sale) {
            // Delete related sell transactions first
            $sale->sellTransactions()->delete();
            
            // If the sale had due amount, handle customer payments
            if ($sale->due_amount > 0 && $sale->customer_id) {
                // Find and reverse any customer payments related to this sale
                $customerPayments = \App\Models\CustomerPayment::where('tenant_id', $sale->tenant_id)
                    ->whereNotNull('sale_ids')
                    ->get();
                
                foreach ($customerPayments as $payment) {
                    $saleIds = json_decode($payment->sale_ids, true);
                    if (is_array($saleIds) && in_array($sale->id, $saleIds)) {
                        // Remove this sale from the payment record
                        $saleIds = array_filter($saleIds, function($id) use ($sale) {
                            return $id != $sale->id;
                        });
                        
                        if (empty($saleIds)) {
                            // If no other sales in this payment, delete the payment record
                            $payment->delete();
                        } else {
                            // Update the payment record with remaining sale IDs
                            $payment->sale_ids = json_encode(array_values($saleIds));
                            $payment->save();
                        }
                    }
                }
            }
        });
    }
}
