<?php
/**
 * Step 1: Check System Requirements
 */

// Check PHP version
$phpVersion = PHP_VERSION;
$phpVersionOk = version_compare($phpVersion, '8.1.0', '>=');

// Check PHP extensions
$requiredExtensions = [
    'openssl' => extension_loaded('openssl'),
    'pdo' => extension_loaded('pdo'),
    'pdo_mysql' => extension_loaded('pdo_mysql'),
    'mbstring' => extension_loaded('mbstring'),
    'tokenizer' => extension_loaded('tokenizer'),
    'xml' => extension_loaded('xml'),
    'ctype' => extension_loaded('ctype'),
    'json' => extension_loaded('json'),
    'bcmath' => extension_loaded('bcmath'),
    'curl' => extension_loaded('curl'),
    'fileinfo' => extension_loaded('fileinfo'),
    'zip' => extension_loaded('zip'),
    'gd' => extension_loaded('gd') || extension_loaded('imagick')
];

// Check directory permissions
$requiredDirectories = [
    'storage/app' => is_writable(__DIR__ . '/../../storage/app'),
    'storage/framework' => is_writable(__DIR__ . '/../../storage/framework'),
    'storage/logs' => is_writable(__DIR__ . '/../../storage/logs'),
    'bootstrap/cache' => is_writable(__DIR__ . '/../../bootstrap/cache'),
    'public' => is_writable(__DIR__ . '/../../public')
];

// Check if composer dependencies are installed
$vendorExists = file_exists(__DIR__ . '/../../vendor/autoload.php');

// Calculate overall status
$extensionsOk = !in_array(false, $requiredExtensions, true);
$permissionsOk = !in_array(false, $requiredDirectories, true);
$allRequirementsMet = $phpVersionOk && $extensionsOk && $permissionsOk && $vendorExists;

?>

<h2>System Requirements Check</h2>
<p style="margin-bottom: 30px; color: #666;">Checking if your server meets all requirements...</p>

<?php if (!$allRequirementsMet): ?>
<div class="alert alert-error">
    <strong>⚠️ Requirements Not Met</strong><br>
    Please fix the issues below before continuing with the installation.
</div>
<?php else: ?>
<div class="alert alert-success">
    <strong>✓ All Requirements Met</strong><br>
    Your server meets all requirements. You can proceed with the installation.
</div>
<?php endif; ?>

<h3 style="margin-top: 30px; margin-bottom: 15px;">PHP Version</h3>
<div class="requirement-check <?php echo $phpVersionOk ? 'pass' : 'fail'; ?>">
    <div>
        <strong>PHP <?php echo $phpVersion; ?></strong>
        <div class="help-text">Required: PHP 8.1.0 or higher</div>
    </div>
    <span class="status-badge <?php echo $phpVersionOk ? 'status-pass' : 'status-fail'; ?>">
        <?php echo $phpVersionOk ? 'PASS' : 'FAIL'; ?>
    </span>
</div>

<h3 style="margin-top: 30px; margin-bottom: 15px;">PHP Extensions</h3>
<?php foreach ($requiredExtensions as $ext => $loaded): ?>
<div class="requirement-check <?php echo $loaded ? 'pass' : 'fail'; ?>">
    <div>
        <strong><?php echo $ext; ?></strong>
        <div class="help-text">Required PHP extension</div>
    </div>
    <span class="status-badge <?php echo $loaded ? 'status-pass' : 'status-fail'; ?>">
        <?php echo $loaded ? 'INSTALLED' : 'MISSING'; ?>
    </span>
</div>
<?php endforeach; ?>

<h3 style="margin-top: 30px; margin-bottom: 15px;">Directory Permissions</h3>
<?php foreach ($requiredDirectories as $dir => $writable): ?>
<div class="requirement-check <?php echo $writable ? 'pass' : 'fail'; ?>">
    <div>
        <strong><?php echo $dir; ?></strong>
        <div class="help-text">Must be writable (chmod 775 or 755)</div>
    </div>
    <span class="status-badge <?php echo $writable ? 'status-pass' : 'status-fail'; ?>">
        <?php echo $writable ? 'WRITABLE' : 'NOT WRITABLE'; ?>
    </span>
</div>
<?php endforeach; ?>

<h3 style="margin-top: 30px; margin-bottom: 15px;">Dependencies</h3>
<div class="requirement-check <?php echo $vendorExists ? 'pass' : 'fail'; ?>">
    <div>
        <strong>Composer Dependencies</strong>
        <div class="help-text">vendor/autoload.php must exist</div>
    </div>
    <span class="status-badge <?php echo $vendorExists ? 'status-pass' : 'status-fail'; ?>">
        <?php echo $vendorExists ? 'INSTALLED' : 'MISSING'; ?>
    </span>
</div>

<?php if (!$vendorExists): ?>
<div class="alert alert-warning" style="margin-top: 20px;">
    <strong>Composer Dependencies Missing</strong><br>
    Run the following command in your terminal:<br>
    <div class="code-block">composer install --no-dev --optimize-autoloader</div>
</div>
<?php endif; ?>

<?php if (!$permissionsOk): ?>
<div class="alert alert-warning" style="margin-top: 20px;">
    <strong>Fix Directory Permissions</strong><br>
    Run the following commands in your terminal:<br>
    <div class="code-block">
chmod -R 775 storage bootstrap/cache<br>
chown -R www-data:www-data storage bootstrap/cache public
    </div>
</div>
<?php endif; ?>

<div class="btn-group">
    <a href="javascript:history.back()" class="btn btn-secondary">← Back</a>
    <?php if ($allRequirementsMet): ?>
        <a href="install.php?step=2" class="btn btn-primary">Continue →</a>
    <?php else: ?>
        <a href="install.php?step=1" class="btn btn-secondary">Recheck Requirements</a>
    <?php endif; ?>
</div>
