<?php
/**
 * Step 4: Run Installation
 */

$installLog = [];
$installationComplete = false;
$installationFailed = false;
$errorMessage = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['start_install'])) {
    
    try {
        $data = $_SESSION['install_data'] ?? [];
        
        if (empty($data)) {
            throw new Exception('Installation data not found. Please start from Step 1.');
        }
        
        // Step 1: Create .env file
        $installLog[] = ['status' => 'running', 'message' => 'Creating environment configuration...'];
        
        $envContent = "APP_NAME=\"{$data['app_name']}\"
APP_ENV=production
APP_KEY=
APP_DEBUG=false
APP_TIMEZONE={$data['app_timezone']}
APP_URL={$data['app_url']}

APP_LOCALE=en
APP_FALLBACK_LOCALE=en
APP_FAKER_LOCALE=en_US

APP_MAINTENANCE_DRIVER=file
APP_MAINTENANCE_STORE=database

BCRYPT_ROUNDS=12

LOG_CHANNEL=stack
LOG_STACK=single
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=error

DB_CONNECTION=mysql
DB_HOST={$data['db_host']}
DB_PORT={$data['db_port']}
DB_DATABASE={$data['db_name']}
DB_USERNAME={$data['db_user']}
DB_PASSWORD={$data['db_password']}

SESSION_DRIVER=database
SESSION_LIFETIME=120
SESSION_ENCRYPT=false
SESSION_PATH=/
SESSION_DOMAIN=null

BROADCAST_CONNECTION=log
FILESYSTEM_DISK=local
QUEUE_CONNECTION=database

CACHE_STORE=database
CACHE_PREFIX=

MEMCACHED_HOST=127.0.0.1

REDIS_CLIENT=phpredis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

MAIL_MAILER=log
MAIL_HOST=127.0.0.1
MAIL_PORT=2525
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS=\"hello@example.com\"
MAIL_FROM_NAME=\"\${APP_NAME}\"

AWS_ACCESS_KEY_ID=
AWS_SECRET_ACCESS_KEY=
AWS_DEFAULT_REGION=us-east-1
AWS_BUCKET=
AWS_USE_PATH_STYLE_ENDPOINT=false

VITE_APP_NAME=\"\${APP_NAME}\"
";
        
        $envPath = __DIR__ . '/../../.env';
        if (file_put_contents($envPath, $envContent) === false) {
            throw new Exception('Failed to create .env file. Check directory permissions.');
        }
        $installLog[0]['status'] = 'success';
        $installLog[0]['message'] = '✓ Environment configuration created';
        
        // Step 2: Generate application key
        $installLog[] = ['status' => 'running', 'message' => 'Generating application key...'];
        
        $output = [];
        $returnVar = 0;
        exec('cd "' . __DIR__ . '/../.." && php artisan key:generate --force 2>&1', $output, $returnVar);
        
        if ($returnVar !== 0) {
            throw new Exception('Failed to generate application key: ' . implode("\n", $output));
        }
        $installLog[1]['status'] = 'success';
        $installLog[1]['message'] = '✓ Application key generated';
        
        // Step 3: Clear configuration cache
        $installLog[] = ['status' => 'running', 'message' => 'Clearing configuration cache...'];
        exec('cd "' . __DIR__ . '/../.." && php artisan config:clear 2>&1', $output, $returnVar);
        $installLog[2]['status'] = 'success';
        $installLog[2]['message'] = '✓ Configuration cache cleared';
        
        // Step 4: Run database migrations
        $installLog[] = ['status' => 'running', 'message' => 'Running database migrations (this may take a moment)...'];
        
        $output = [];
        exec('cd "' . __DIR__ . '/../.." && php artisan migrate --force 2>&1', $output, $returnVar);
        
        if ($returnVar !== 0) {
            $migrationError = implode("\n", $output);
            if (strpos($migrationError, 'SQLSTATE') !== false || strpos($migrationError, 'error') !== false) {
                throw new Exception('Database migration failed: ' . $migrationError);
            }
        }
        $installLog[3]['status'] = 'success';
        $installLog[3]['message'] = '✓ Database migrations completed';
        
        // Step 5: Create admin user
        $installLog[] = ['status' => 'running', 'message' => 'Creating administrator account...'];
        
        // Connect to database and create admin user
        $pdo = new PDO(
            "mysql:host={$data['db_host']};port={$data['db_port']};dbname={$data['db_name']}",
            $data['db_user'],
            $data['db_password'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        $passwordHash = password_hash($data['admin_password'], PASSWORD_BCRYPT);
        
        // Check which columns exist in the users table
        $allColumns = $pdo->query("SHOW COLUMNS FROM users")->fetchAll(PDO::FETCH_COLUMN);
        $hasPhoneColumn = in_array('phone', $allColumns);
        $hasTenantColumn = in_array('tenant_id', $allColumns);
        
        $tenantId = null;
        
        // If tenant_id column exists, we need to handle the tenant relationship
        if ($hasTenantColumn) {
            // Check if tenants table exists
            $tables = $pdo->query("SHOW TABLES LIKE 'tenants'")->fetchAll();
            if (count($tables) > 0) {
                // Tenants table exists, check if we need to create a tenant
                $existingTenant = $pdo->query("SELECT id FROM tenants ORDER BY id ASC LIMIT 1")->fetch(PDO::FETCH_ASSOC);
                
                if ($existingTenant) {
                    // Use existing tenant
                    $tenantId = $existingTenant['id'];
                } else {
                    // Create a default tenant
                    $tenantColumns = $pdo->query("SHOW COLUMNS FROM tenants")->fetchAll(PDO::FETCH_COLUMN);
                    
                    // Build tenant insert based on available columns
                    $tenantInsertCols = ['created_at', 'updated_at'];
                    $tenantInsertVals = ['NOW()', 'NOW()'];
                    $tenantData = [];
                    
                    // Add name if column exists
                    if (in_array('name', $tenantColumns)) {
                        $tenantInsertCols[] = 'name';
                        $tenantInsertVals[] = '?';
                        $tenantData[] = $data['app_name'] ?? 'Main Business';
                    }
                    
                    // Add domain if column exists (required field)
                    if (in_array('domain', $tenantColumns)) {
                        $tenantInsertCols[] = 'domain';
                        $tenantInsertVals[] = '?';
                        // Extract domain from APP_URL or use a default
                        $domain = parse_url($data['app_url'], PHP_URL_HOST) ?? 'localhost';
                        $tenantData[] = $domain;
                    }
                    
                    $tenantSql = "INSERT INTO tenants (" . implode(', ', $tenantInsertCols) . ") VALUES (" . implode(', ', $tenantInsertVals) . ")";
                    $stmt = $pdo->prepare($tenantSql);
                    $stmt->execute($tenantData);
                    $tenantId = $pdo->lastInsertId();
                }
            } else {
                // Tenants table doesn't exist but tenant_id column does - set to 1
                $tenantId = 1;
            }
        }
        
        // Build INSERT query based on available columns
        $columns = ['name', 'email', 'password', 'created_at', 'updated_at'];
        $values = [$data['admin_name'], $data['admin_email'], $passwordHash];
        $placeholders = ['?', '?', '?', 'NOW()', 'NOW()'];
        
        // Add tenant_id if column exists
        if ($hasTenantColumn && $tenantId) {
            array_splice($columns, 0, 0, ['tenant_id']);
            array_splice($values, 0, 0, [$tenantId]);
            array_splice($placeholders, 0, 0, ['?']);
        }
        
        // Add phone if column exists and data provided
        if ($hasPhoneColumn) {
            $columns[] = 'phone';
            $values[] = $data['admin_phone'] ?? null;
            $placeholders[] = '?';
        }
        
        // Execute the insert
        $sql = "INSERT INTO users (" . implode(', ', $columns) . ") VALUES (" . implode(', ', $placeholders) . ")";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($values);
        $userId = $pdo->lastInsertId();
        
        // Assign Super Admin role if roles system exists
        try {
            $rolesTable = $pdo->query("SHOW TABLES LIKE 'roles'")->fetchAll();
            if (count($rolesTable) > 0) {
                // Create super-admin role if it doesn't exist
                $superAdminRole = $pdo->query("SELECT id FROM roles WHERE name = 'super-admin' LIMIT 1")->fetch(PDO::FETCH_ASSOC);
                
                if (!$superAdminRole) {
                    // Create super-admin role
                    $pdo->exec("INSERT INTO roles (name, guard_name, created_at, updated_at) VALUES ('super-admin', 'web', NOW(), NOW())");
                    $roleId = $pdo->lastInsertId();
                } else {
                    $roleId = $superAdminRole['id'];
                }
                
                // Check if model_has_roles table exists (Spatie Laravel Permission)
                $modelHasRolesTable = $pdo->query("SHOW TABLES LIKE 'model_has_roles'")->fetchAll();
                if (count($modelHasRolesTable) > 0) {
                    // Assign role to user (use prepared statement for safety)
                    $stmt = $pdo->prepare("INSERT IGNORE INTO model_has_roles (role_id, model_type, model_id) VALUES (?, ?, ?)");
                    $stmt->execute([$roleId, 'App\\Models\\User', $userId]);
                }
                
                // Give super-admin ALL permissions
                $permissionsTable = $pdo->query("SHOW TABLES LIKE 'permissions'")->fetchAll();
                if (count($permissionsTable) > 0) {
                    $roleHasPermissionsTable = $pdo->query("SHOW TABLES LIKE 'role_has_permissions'")->fetchAll();
                    if (count($roleHasPermissionsTable) > 0) {
                        // Get all permissions
                        $allPermissions = $pdo->query("SELECT id FROM permissions")->fetchAll(PDO::FETCH_COLUMN);
                        
                        // Assign all permissions to super-admin role
                        if (!empty($allPermissions)) {
                            $stmt = $pdo->prepare("INSERT IGNORE INTO role_has_permissions (permission_id, role_id) VALUES (?, ?)");
                            foreach ($allPermissions as $permissionId) {
                                $stmt->execute([$permissionId, $roleId]);
                            }
                        }
                    }
                }
            }
        } catch (Exception $roleException) {
            // Role assignment failed, but user was created - continue
            // Don't throw error, just log it silently
        }
        
        // Create active subscription for tenant (bypass subscription requirement)
        try {
            $subscriptionsTable = $pdo->query("SHOW TABLES LIKE 'subscriptions'")->fetchAll();
            if ($tenantId && count($subscriptionsTable) > 0) {
                // Check if plans table exists
                $plansTable = $pdo->query("SHOW TABLES LIKE 'plans'")->fetchAll();
                if (count($plansTable) > 0) {
                    // Get or create a default plan
                    $plan = $pdo->query("SELECT id FROM plans ORDER BY id ASC LIMIT 1")->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$plan) {
                        // Create a lifetime plan
                        $pdo->exec("INSERT INTO plans (name, price, duration_months, features, created_at, updated_at) VALUES ('Lifetime Plan', 0, 999, 'Unlimited Access - Full Features', NOW(), NOW())");
                        $planId = $pdo->lastInsertId();
                    } else {
                        $planId = $plan['id'];
                    }
                    
                    // Check if subscription already exists
                    $existingSub = $pdo->query("SELECT id FROM subscriptions WHERE tenant_id = {$tenantId} AND status = 'active'")->fetch();
                    
                    if (!$existingSub) {
                        // Create lifetime active subscription (100 years)
                        $pdo->exec("INSERT INTO subscriptions (tenant_id, plan_id, status, start_date, end_date, created_at, updated_at) VALUES ({$tenantId}, {$planId}, 'active', NOW(), DATE_ADD(NOW(), INTERVAL 100 YEAR), NOW(), NOW())");
                    }
                }
            }
        } catch (Exception $subException) {
            // Subscription creation failed, but user was created - continue
        }
        
        $installLog[4]['status'] = 'success';
        $installLog[4]['message'] = '✓ Administrator account created with Super Admin privileges';
        
        // Step 6: Create storage symlink
        $installLog[] = ['status' => 'running', 'message' => 'Creating storage symlink...'];
        exec('cd "' . __DIR__ . '/../.." && php artisan storage:link 2>&1', $output, $returnVar);
        $installLog[5]['status'] = 'success';
        $installLog[5]['message'] = '✓ Storage symlink created';
        
        // Step 7: Clear all caches
        $installLog[] = ['status' => 'running', 'message' => 'Clearing application caches...'];
        exec('cd "' . __DIR__ . '/../.." && php artisan route:clear 2>&1', $output, $returnVar);
        exec('cd "' . __DIR__ . '/../.." && php artisan view:clear 2>&1', $output, $returnVar);
        exec('cd "' . __DIR__ . '/../.." && php artisan cache:clear 2>&1', $output, $returnVar);
        $installLog[6]['status'] = 'success';
        $installLog[6]['message'] = '✓ Application caches cleared';
        
        // Step 8: Optimize application
        $installLog[] = ['status' => 'running', 'message' => 'Optimizing application...'];
        exec('cd "' . __DIR__ . '/../.." && php artisan optimize 2>&1', $output, $returnVar);
        $installLog[7]['status'] = 'success';
        $installLog[7]['message'] = '✓ Application optimized';
        
        // Step 9: Set proper permissions
        $installLog[] = ['status' => 'running', 'message' => 'Setting directory permissions...'];
        
        @chmod(__DIR__ . '/../../storage', 0775);
        @chmod(__DIR__ . '/../../bootstrap/cache', 0775);
        
        $installLog[8]['status'] = 'success';
        $installLog[8]['message'] = '✓ Permissions configured';
        
        $installationComplete = true;
        
    } catch (Exception $e) {
        $installationFailed = true;
        $errorMessage = $e->getMessage();
        
        // Mark current step as failed
        if (!empty($installLog)) {
            $lastIndex = count($installLog) - 1;
            $installLog[$lastIndex]['status'] = 'error';
            $installLog[$lastIndex]['message'] = '✗ ' . $errorMessage;
        }
    }
}
?>

<h2>Installation Progress</h2>
<p style="margin-bottom: 30px; color: #666;">
    <?php if (!$installationComplete && !$installationFailed): ?>
        Click "Start Installation" to begin the setup process.
    <?php elseif ($installationComplete): ?>
        Installation completed successfully!
    <?php else: ?>
        Installation encountered an error.
    <?php endif; ?>
</p>

<?php if ($installationFailed): ?>
<div class="alert alert-error">
    <strong>⚠️ Installation Failed</strong><br>
    <?php echo nl2br(htmlspecialchars($errorMessage)); ?>
    <br><br>
    <strong>Common Solutions:</strong>
    <ul style="margin: 10px 0 0 20px;">
        <li>Check database credentials are correct</li>
        <li>Ensure database user has CREATE, ALTER, DROP privileges</li>
        <li>Verify storage and bootstrap/cache directories are writable</li>
        <li>Check PHP version is 8.1 or higher</li>
        <li>Review server error logs for details</li>
    </ul>
</div>
<?php endif; ?>

<?php if (!empty($installLog)): ?>
<div style="background: #f9f9f9; border-radius: 6px; padding: 20px; margin: 20px 0;">
    <h3 style="margin-bottom: 15px;">Installation Log</h3>
    <?php foreach ($installLog as $log): ?>
        <div style="padding: 10px; margin-bottom: 10px; background: white; border-radius: 4px; border-left: 4px solid <?php echo $log['status'] === 'success' ? '#28a745' : ($log['status'] === 'error' ? '#dc3545' : '#ffc107'); ?>;">
            <?php if ($log['status'] === 'running'): ?>
                <span class="loading" style="width: 16px; height: 16px; margin-right: 10px;"></span>
            <?php endif; ?>
            <?php echo htmlspecialchars($log['message']); ?>
        </div>
    <?php endforeach; ?>
</div>
<?php endif; ?>

<?php if ($installationComplete): ?>
<div class="alert alert-success">
    <strong>✓ Installation Complete!</strong><br>
    All steps completed successfully. Your POS system is ready to use.
</div>
<?php endif; ?>

<form method="POST" action="install.php?step=4">
    <div class="btn-group">
        <?php if (!$installationComplete && !$installationFailed): ?>
            <a href="install.php?step=3" class="btn btn-secondary">← Back</a>
            <button type="submit" name="start_install" class="btn btn-primary">🚀 Start Installation</button>
        <?php elseif ($installationFailed): ?>
            <a href="install.php?step=2" class="btn btn-secondary">← Back to Database Settings</a>
            <button type="submit" name="start_install" class="btn btn-primary">🔄 Retry Installation</button>
        <?php else: ?>
            <span></span>
            <a href="install.php?step=5" class="btn btn-primary">Continue →</a>
        <?php endif; ?>
    </div>
</form>

<?php if (!$installationComplete && !$installationFailed): ?>
<div class="alert alert-warning" style="margin-top: 30px;">
    <strong>⏱️ Please Wait</strong><br>
    The installation process may take 1-2 minutes. Do not close this page or refresh your browser.
</div>
<?php endif; ?>
