<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use App\Models\Product;
use App\Models\Payment;
use App\Models\Setting;
use Illuminate\Http\Request;

class SaleController extends Controller
{
    public function create()
    {
        $tenant = app('tenant');
        // Fetch products and customers for the form
        $products = Product::where('tenant_id', $tenant->id)->get();
        $customers = \App\Models\Customer::where('tenant_id', $tenant->id)->get();
        return view('sales.create', compact('products', 'customers'));
    }

    public function invoice($id)
    {
        $tenant = app('tenant');
        $sale = Sale::with(['sellTransactions.product', 'payments', 'customer', 'exchanges.brand', 'exchanges.mobileModel'])
            ->where('tenant_id', $tenant->id)
            ->findOrFail($id);
        $all_settings = Setting::where('tenant_id', $tenant->id)->first();

        return view('sell.invoice', compact('sale', 'all_settings'));
    }

    public function ajaxInvoice($id)
    {
        try {
            $tenant = app('tenant');
            $sale = Sale::with(['sellTransactions.product', 'payments', 'customer', 'exchanges.brand', 'exchanges.mobileModel'])
                ->where('tenant_id', $tenant->id)
                ->findOrFail($id);
            $all_settings = Setting::where('tenant_id', $tenant->id)->first();

            $html = view('sell.invoice', compact('sale', 'all_settings'))->render();
            

            return response()->json(['html' => $html]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to load invoice: ' . $e->getMessage()], 500);
        }
    }

    public function tableInvoice($id)
    {
        try {
            $tenant = app('tenant');
            $sale = Sale::with(['sellTransactions.product', 'payments', 'customer', 'exchanges.brand', 'exchanges.mobileModel'])
                ->where('tenant_id', $tenant->id)
                ->findOrFail($id);
            $all_settings = Setting::where('tenant_id', $tenant->id)->first();

            return view('sell.invoice-table', compact('sale', 'all_settings'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to load invoice: ' . $e->getMessage());
        }
    }

    public function store(Request $request)
    {
        return $request;
        // Validate the request
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'products' => 'required|array',
            'payment_method' => 'required|in:cash,credit',
        ]);

        $tenant = app('tenant');

        // Create Sale
        $sale = Sale::create([
            'tenant_id' => $tenant->id,
            'customer_id' => $request->customer_id,
            'total_price' => array_sum(array_column($request->products, 'price')), // Calculate total price from products
            'payment_method' => $request->payment_method,
        ]);

        // Attach products to sale
        foreach ($request->products as $product) {
            $sale->products()->attach($product['id'], [
                'quantity' => $product['quantity'],
                'price' => $product['price'],
            ]);
        }

        // Process payment
        Payment::create([
            'tenant_id' => $tenant->id,
            'sale_id' => $sale->id,
            'payment_type' => $request->payment_method,
            'amount' => $sale->total_price,
        ]);

        return redirect()->route('sales.create')->with('success', 'Sale created successfully!');
    }

    public function edit(Sale $sale)
    {
        $tenant = app('tenant');
        $sale = Sale::where('tenant_id', $tenant->id)->findOrFail($sale->id);
        // Fetch products and customers for the form
        $products = Product::where('tenant_id', $tenant->id)->get();
        $customers = \App\Models\Customer::where('tenant_id', $tenant->id)->get();
        return view('sales.edit', compact('sale', 'products', 'customers'));
    }

    public function update(Request $request, Sale $sale)
    {
        $tenant = app('tenant');
        $sale = Sale::where('tenant_id', $tenant->id)->findOrFail($sale->id);

        // Validate the request
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'products' => 'required|array',
            'payment_method' => 'required|in:cash,credit',
        ]);

        // Update Sale
        $sale->update([
            'customer_id' => $request->customer_id,
            'total_price' => array_sum(array_column($request->products, 'price')), // Calculate total price from products
            'payment_method' => $request->payment_method,
        ]);

        // Update the attached products
        $sale->products()->detach(); // Remove all previous products
        foreach ($request->products as $product) {
            $sale->products()->attach($product['id'], [
                'quantity' => $product['quantity'],
                'price' => $product['price'],
            ]);
        }

        // Process payment update
        $sale->payments()->delete(); // Remove previous payments
        Payment::create([
            'tenant_id' => $tenant->id,
            'sale_id' => $sale->id,
            'payment_type' => $request->payment_method,
            'amount' => $sale->total_price,
        ]);

        return redirect()->route('sales.edit', $sale)->with('success', 'Sale updated successfully!');
    }

    public function show($id, Request $request = null)
    {
        $tenant = app('tenant');
        $sale = Sale::with(['customer', 'sellTransactions.product.brand', 'payments', 'exchanges.brand', 'exchanges.mobileModel'])
            ->where('tenant_id', $tenant->id)
            ->findOrFail($id);

        // Transform sellTransactions to items for the view
        $sale->items = $sale->sellTransactions->map(function($transaction) {
            return (object) [
                'product_name' => $transaction->product ? $transaction->product->name : 'Unknown Product',
                'product' => $transaction->product,
                'quantity' => $transaction->sold_quantity,
                'unit_price' => $transaction->sell_price,
                'total_price' => $transaction->sell_price * $transaction->sold_quantity,
            ];
        });

        // Calculate totals
        $sale->subtotal = $sale->items->sum('total_price');
        $sale->tax_amount = $sale->tax_amount ?? 0;
        $sale->discount_amount = $sale->discount_amount ?? 0;

        // If this is an API request, return JSON
        if ($request && ($request->expectsJson() || $request->is('api/*'))) {
            return response()->json($sale);
        }

        return view('sales.show', compact('sale'));
    }

    public function index()
    {
        $tenant = app('tenant');
        
        // Get today's sales data
        $today = now()->startOfDay();
        $todaySales = Sale::where('tenant_id', $tenant->id)
            ->where('created_at', '>=', $today)
            ->get();
        
        // Calculate statistics
        $totalSales = $todaySales->count();
        $todayRevenue = $todaySales->sum('total_amount');
        $todayProfit = $todaySales->sum('profit_amount') ?? $todaySales->sum(function($sale) {
            return $sale->sellTransactions->sum('profit');
        });
        $pendingPayments = Sale::where('tenant_id', $tenant->id)
            ->whereIn('payment_status', ['unpaid', 'partial'])
            ->count();
        
        // Get recent sales
        $recentSales = Sale::with(['customer', 'sellTransactions'])
            ->where('tenant_id', $tenant->id)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Payment status counts
        $paidCount = Sale::where('tenant_id', $tenant->id)->where('payment_status', 'paid')->count();
        $unpaidCount = Sale::where('tenant_id', $tenant->id)->where('payment_status', 'unpaid')->count();
        $partialCount = Sale::where('tenant_id', $tenant->id)->where('payment_status', 'partial')->count();
        
        // Weekly sales data for chart
        $weeklyData = [];
        $weeklyLabels = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $weeklyLabels[] = $date->format('M d');
            $dayTotal = Sale::where('tenant_id', $tenant->id)
                ->whereDate('created_at', $date)
                ->sum('total_amount');
            $weeklyData[] = $dayTotal;
        }
        
        return view('sales.index', compact(
            'totalSales', 'todayRevenue', 'todayProfit', 'pendingPayments',
            'recentSales', 'paidCount', 'unpaidCount', 'partialCount',
            'weeklyData', 'weeklyLabels'
        ));
    }

    public function history()
    {
        $tenant = app('tenant');
        
        // Load regular sales with related data
        $regularSales = Sale::with(['customer', 'sellTransactions', 'exchanges.brand', 'exchanges.mobileModel'])
            ->where('tenant_id', $tenant->id)
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Load direct sales
        $directSales = \App\Models\DirectSales::with('Customer')
            ->orderBy('direct_date_time', 'desc')
            ->get();
        
        // Transform direct sales to match regular sales structure
        $transformedDirectSales = $directSales->map(function($directSale) {
            return (object) [
                'id' => 'DS-' . $directSale->id, // Prefix to distinguish direct sales
                'type' => 'direct_sale',
                'created_at' => $directSale->direct_date_time ? \Carbon\Carbon::parse($directSale->direct_date_time) : now(),
                'customer' => $directSale->Customer,
                'customer_name' => $directSale->direct_customer_name,
                'total_amount' => $directSale->direct_sell_price * $directSale->direct_sold_quantity,
                'net_amount' => $directSale->direct_sell_price * $directSale->direct_sold_quantity,
                'due_amount' => 0, // Direct sales are typically fully paid
                'payment_method' => 'cash', // Default for direct sales
                'payment_status' => 'paid', // Direct sales are typically paid
                'product_type' => $directSale->product_type,
                'quantity' => $directSale->direct_sold_quantity,
                'profit' => $directSale->direct_profit,
                'sellTransactions' => collect([
                    (object) [
                        'product_name' => $directSale->product_type,
                        'sold_quantity' => $directSale->direct_sold_quantity,
                        'sell_price' => $directSale->direct_sell_price,
                        'buy_price' => $directSale->direct_cost_price,
                    ]
                ]),
                'exchanges' => collect(),
                'original_model' => $directSale, // Keep reference to original model
            ];
        });
        
        // Transform regular sales to match the structure
        $transformedRegularSales = $regularSales->map(function($sale) {
            $sale->type = 'regular_sale';
            $sale->original_model = $sale;
            // Add exchange information
            $sale->has_exchanges = $sale->exchanges->count() > 0;
            $sale->exchange_count = $sale->exchanges->count();
            $sale->total_exchange_value = $sale->exchanges->sum('price');
            return $sale;
        });
        
        // Merge and sort all sales by date
        $allSales = $transformedRegularSales->concat($transformedDirectSales)
            ->sortByDesc('created_at')
            ->values();
        
        // Manual pagination
        $currentPage = request()->get('page', 1);
        $perPage = 25;
        $total = $allSales->count();
        $offset = ($currentPage - 1) * $perPage;
        
        $paginatedSales = $allSales->slice($offset, $perPage);
        
        // Create pagination instance
        $sales = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedSales,
            $total,
            $perPage,
            $currentPage,
            [
                'path' => request()->url(),
                'pageName' => 'page',
            ]
        );
        
        return view('sales.history', compact('sales'));
    }

    public function markAsPaid($id)
    {
        try {
            $tenant = app('tenant');
            $sale = Sale::where('tenant_id', $tenant->id)->findOrFail($id);
            
            $sale->update([
                'payment_status' => 'paid',
                'due_amount' => 0
            ]);
            
            return response()->json(['success' => true, 'message' => 'Sale marked as paid successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error updating sale status'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $tenant = app('tenant');
            $sale = Sale::where('tenant_id', $tenant->id)->findOrFail($id);
            
            // Use the model's built-in deletion logic which handles cleanup via booted() method
            $sale->delete();
            
            return response()->json(['success' => true, 'message' => 'Sale deleted successfully']);
        } catch (\Exception $e) {
            \Log::error('Sale deletion error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Error deleting sale: ' . $e->getMessage()], 500);
        }
    }

    public function getItems($id)
    {
        try {
            $tenant = app('tenant');
            $sale = Sale::with(['sellTransactions.product', 'exchanges.brand', 'exchanges.mobileModel'])
                ->where('tenant_id', $tenant->id)
                ->findOrFail($id);

            $transactions = $sale->sellTransactions->map(function($transaction) {
                return [
                    'id' => $transaction->id,
                    'product_name' => $transaction->product ? $transaction->product->name : 'N/A',
                    'brand' => $transaction->product ? $transaction->product->brand : 'N/A',
                    'model' => $transaction->product ? $transaction->product->model : 'N/A',
                    'imei_or_sn' => $transaction->product ? $transaction->product->imei_or_sn : 'N/A',
                    'sold_quantity' => $transaction->sold_quantity,
                    'sell_price' => $transaction->sell_price,
                    'buy_price' => $transaction->product ? $transaction->product->buy_price : 0,
                    'warranty' => $transaction->warranty ?? 'No warranty',
                ];
            });

            $exchanges = $sale->exchanges->map(function($exchange) {
                return [
                    'id' => $exchange->id,
                    'brand_name' => $exchange->brand ? $exchange->brand->name : null,
                    'custom_brand' => $exchange->custom_brand,
                    'model_name' => $exchange->mobileModel ? $exchange->mobileModel->name : null,
                    'custom_model' => $exchange->custom_model,
                    'imei' => $exchange->imei,
                    'price' => $exchange->price,
                ];
            });

            return response()->json([
                'success' => true,
                'transactions' => $transactions,
                'exchanges' => $exchanges,
                'sale_id' => $sale->id
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error loading sale items'], 500);
        }
    }
}
