<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class SubscriptionController extends Controller
{
    /**
     * Display a listing of subscriptions for the tenant.
     */
    public function index()
    {
        $subscriptions = Subscription::with(['plan', 'tenant'])->get();
        return view('admin.subscriptions.index', compact('subscriptions'));
    }

    /**
     * Show the form for creating a new subscription.
     */
    public function create()
    {
        $tenant = app('tenant');
        $existingSubscription = Subscription::where('tenant_id', $tenant->id)
            ->where('status', '!=', 'active')
            ->latest()
            ->first();

        $plans = Plan::where('is_active', true)->get();
        return view('admin.subscriptions.create', compact('plans', 'existingSubscription'));
    }

    /**
     * Store a newly created subscription in storage.
     */
    public function store(Request $request)
    {
        $tenant = app('tenant');

        // Check if there is already a pending subscription
        $pendingSubscription = Subscription::where('tenant_id', $tenant->id)
            ->where('status', 'pending')
            ->first();

        if ($pendingSubscription) {
            return redirect()->back()->withErrors('You already have a subscription request pending approval. Please wait for admin approval.');
        }

        $validator = Validator::make($request->all(), [
            'plan_id' => 'required|exists:plans,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $plan = Plan::findOrFail($request->plan_id);

        Subscription::create([
            'tenant_id' => $tenant->id,
            'plan_id' => $plan->id,
            'status' => 'pending',
            'start_date' => now(),
            'end_date' => null,
            'price' => $plan->price,
            'interval' => $plan->interval,
        ]);

        return redirect()->route('dashboard')->with('info', 'Subscription request submitted. Please wait for admin approval.');
    }

    /**
     * Display the specified subscription.
     */
    public function show($id)
    {
        $subscription = Subscription::with(['plan', 'tenant'])->findOrFail($id);
        return view('admin.subscriptions.show', compact('subscription'));
    }

    /**
     * Show the form for editing the specified subscription.
     */
    public function edit($id)
    {
        $subscription = Subscription::findOrFail($id);
        $plans = Plan::where('is_active', true)->get();
        return view('admin.subscriptions.edit', compact('subscription', 'plans'));
    }

    /**
     * Update the specified subscription in storage.
     */
    public function update(Request $request, $id)
    {
        $subscription = Subscription::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'plan_id' => 'required|exists:plans,id',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'required|in:active,cancelled,expired',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $plan = Plan::findOrFail($request->plan_id);

        $subscription->update([
            'plan_id' => $plan->id,
            'status' => $request->status,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'price' => $plan->price,
            'interval' => $plan->interval,
        ]);

        return redirect()->route('subscriptions.index')->with('success', 'Subscription updated successfully.');
    }

    /**
     * Approve the specified subscription.
     */
    public function approve($id)
    {
        $subscription = Subscription::findOrFail($id);
        $subscription->update(['status' => 'active']);

        return redirect()->route('subscriptions.index')->with('success', 'Subscription approved successfully.');
    }

    /**
     * Remove the specified subscription from storage.
     */
    public function destroy($id)
    {
        $subscription = Subscription::findOrFail($id);
        $subscription->delete();

        return redirect()->route('subscriptions.index')->with('success', 'Subscription deleted successfully.');
    }
}
