<?php

namespace App\Services;

use App\Models\Setting;
use Exception;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected $settings;
    
    public function __construct()
    {
        $tenant = app('tenant');
        $this->settings = Setting::where('tenant_id', $tenant->id)->first();
    }
    
    /**
     * Format phone number with country code
     */
    public function formatPhoneNumber($phone)
    {
        if (!$this->settings || !$this->settings->whatsapp_auto_format_numbers) {
            return $phone;
        }
        
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // If already starts with country code, return as is
        if (strlen($phone) > 10 && (str_starts_with($phone, '880') || str_starts_with($phone, '91') || str_starts_with($phone, '1'))) {
            return $phone;
        }
        
        // Add default country code
        $countryCode = $this->settings->whatsapp_country_code ?? '+880';
        $countryCode = str_replace('+', '', $countryCode);
        
        // Remove leading zero if exists (common in Bangladesh)
        if (str_starts_with($phone, '0')) {
            $phone = substr($phone, 1);
        }
        
        return $countryCode . $phone;
    }
    
    /**
     * Replace message variables with actual data
     */
    public function replaceMessageVariables($template, $data = [])
    {
        $defaultData = [
            'company_name' => $this->settings->company_name ?? 'Your Company',
            'customer_name' => $data['customer_name'] ?? 'Valued Customer',
            'customer_phone' => $data['customer_phone'] ?? '',
            'invoice_number' => $data['invoice_number'] ?? 'INV-000',
            'invoice_date' => $data['invoice_date'] ?? date('M d, Y'),
            'due_date' => $data['due_date'] ?? date('M d, Y', strtotime('+30 days')),
            'item_list' => $data['item_list'] ?? 'Items purchased',
            'currency' => $this->settings->currency ?? 'BDT',
            'total_amount' => $data['total_amount'] ?? '0.00',
            'payment_method' => $data['payment_method'] ?? 'Cash',
            'payment_status' => $data['payment_status'] ?? 'Paid',
            'invoice_link' => $data['invoice_link'] ?? url('/'),
        ];
        
        // Merge with provided data
        $data = array_merge($defaultData, $data);
        
        // Replace all variables
        foreach ($data as $key => $value) {
            $template = str_replace('{' . $key . '}', $value, $template);
        }
        
        // Add business hours if enabled
        if ($this->settings && $this->settings->whatsapp_include_business_hours && $this->settings->whatsapp_business_hours) {
            $template .= "\n\n📞 *Business Hours:* " . $this->settings->whatsapp_business_hours;
        }
        
        return $template;
    }
    
    /**
     * Generate WhatsApp URL for QR code or direct link
     */
    public function generateWhatsAppUrl($phone, $message)
    {
        // Format phone number
        $formattedPhone = $this->formatPhoneNumber($phone);
        
        // Encode message for URL
        $encodedMessage = urlencode($message);
        
        // Generate WhatsApp URL using official WhatsApp API
        return "https://api.whatsapp.com/send/?phone={$formattedPhone}&text={$encodedMessage}";
    }
    
    /**
     * Generate QR code URL using Google Charts API
     */
    public function generateQrCodeUrl($phone, $message, $size = 300)
    {
        $whatsappUrl = $this->generateWhatsAppUrl($phone, $message);
        $encodedUrl = urlencode($whatsappUrl);
        
        return "https://chart.googleapis.com/chart?chs={$size}x{$size}&cht=qr&chl={$encodedUrl}&choe=UTF-8";
    }
    
    /**
     * Generate QR code using QR Server API (alternative)
     */
    public function generateQrCodeUrlAlt($phone, $message, $size = 300)
    {
        $whatsappUrl = $this->generateWhatsAppUrl($phone, $message);
        $encodedUrl = urlencode($whatsappUrl);
        
        return "https://api.qrserver.com/v1/create-qr-code/?size={$size}x{$size}&data={$encodedUrl}";
    }
    
    /**
     * Create WhatsApp data for invoice
     */
    public function createInvoiceWhatsApp($phone, $invoiceData)
    {
        $template = $this->settings->whatsapp_invoice_message_template ?? $this->getDefaultInvoiceTemplate();
        $message = $this->replaceMessageVariables($template, $invoiceData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for welcome message
     */
    public function createWelcomeWhatsApp($phone, $customerData)
    {
        $template = $this->settings->whatsapp_welcome_message ?? $this->getDefaultWelcomeTemplate();
        $message = $this->replaceMessageVariables($template, $customerData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for payment reminder
     */
    public function createPaymentReminderWhatsApp($phone, $invoiceData)
    {
        $template = $this->settings->whatsapp_payment_reminder_message ?? $this->getDefaultReminderTemplate();
        $message = $this->replaceMessageVariables($template, $invoiceData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for delivery notification
     */
    public function createDeliveryWhatsApp($phone, $orderData)
    {
        $template = $this->settings->whatsapp_delivery_message ?? $this->getDefaultDeliveryTemplate();
        $message = $this->replaceMessageVariables($template, $orderData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for thank you message
     */
    public function createThankYouWhatsApp($phone, $purchaseData)
    {
        $template = $this->settings->whatsapp_thank_you_message ?? $this->getDefaultThankYouTemplate();
        $message = $this->replaceMessageVariables($template, $purchaseData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Default templates
     */
    private function getDefaultInvoiceTemplate()
    {
        return "🧾 *Invoice #{invoice_number}*\n\n" .
               "👤 *Customer:* {customer_name}\n" .
               "📞 *Phone:* {customer_phone}\n" .
               "📅 *Date:* {invoice_date}\n\n" .
               "📦 *Items:*\n{item_list}\n\n" .
               "💰 *Total Amount:* {currency} {total_amount}\n" .
               "💳 *Payment Method:* {payment_method}\n" .
               "📊 *Payment Status:* {payment_status}\n\n" .
               "🙏 *Thank you for your business!*\n\n" .
               "🔗 *View Invoice:* {invoice_link}";
    }
    
    private function getDefaultWelcomeTemplate()
    {
        return "🎉 *Welcome to {company_name}!*\n\n" .
               "Hello {customer_name}! 👋\n\n" .
               "Thank you for choosing us for your mobile phone needs.\n\n" .
               "📱 For any assistance, feel free to contact us.\n\n" .
               "Best regards,\n{company_name} Team 🙏";
    }
    
    private function getDefaultReminderTemplate()
    {
        return "💳 *Payment Reminder*\n\n" .
               "Dear {customer_name},\n\n" .
               "This is a friendly reminder that your invoice #{invoice_number} for {currency} {total_amount} is due.\n\n" .
               "📅 *Due Date:* {due_date}\n\n" .
               "Please make the payment at your earliest convenience.\n\n" .
               "🔗 *View Invoice:* {invoice_link}\n\n" .
               "Thank you! 🙏";
    }
    
    private function getDefaultDeliveryTemplate()
    {
        return "🚚 *Order Delivery Update*\n\n" .
               "Dear {customer_name},\n\n" .
               "Your order #{invoice_number} is ready for delivery! 📦\n\n" .
               "📱 *Items:*\n{item_list}\n\n" .
               "Our delivery team will contact you shortly.\n\n" .
               "Thank you! 🙏";
    }
    
    private function getDefaultThankYouTemplate()
    {
        return "🎉 *Thank You for Your Purchase!*\n\n" .
               "Dear {customer_name},\n\n" .
               "Thank you for choosing {company_name}! ✅\n\n" .
               "🧾 *Invoice:* #{invoice_number}\n" .
               "💰 *Amount:* {currency} {total_amount}\n\n" .
               "We hope you enjoy your new device! 📱\n\n" .
               "Best regards,\n{company_name} Team 🙏";
    }
}